<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\Tracking;
use App\Product;
use App\Category;
use App\Setting;
use App\User;
use Carbon\Carbon;

class TrackingController extends Controller
{

  public function index(Request $request) {
    $limit = $request->query('limit') ? (int) $request->query('limit') : 20;

    if (app('auth')->user()->hasRole('Superadmin')) {
      $data = Tracking::with(['creator', 'category', 'product']); 

      if($request->query('user_id')) {
        $data = $data->where('created_by', $request->query('user_id'));
      }

    } else {
      $data = Tracking::with(['category', 'product'])->where('created_by', app('auth')->user()->id); 
      
    }

    if($request->query('from_date') && $request->query('end_date')) {
      $start = Carbon::createFromFormat('Y-m-d', $request->query('from_date'));
      $end = Carbon::createFromFormat('Y-m-d', $request->query('end_date'));
      $data = $data->whereBetween('created_at', [$start->startOfDay()->format('Y-m-d H:i:s'), $end->endOfDay()->format('Y-m-d H:i:s')]);
    }

    if($request->query('product_id')) {
      $data = $data->where('product_id', $request->query('product_id'));
    }

    if($request->query('category_id')) {
      $data = $data->where('category_id', $request->query('category_id'));
    }
    
    $data = $data->orderBy('created_at', 'desc')->paginate($limit);
    return response()->json($data);
  }

  public function store(Request $request, $referal_code, $product_id, $category_id = null) {

    $ip= $request->ip();
    // $ip = "175.143.31.41";

    // echo $referal_code." ".$product_id." ".$category_id;
    $user_id = null;
    $pid = null;
    $cid = null;

    if ($referal_code) {
      $user = User::where('referal_code', $referal_code)
        ->where('status', true)->first();

      if ($user) {
        $user_id = $user->id;
      }
    }

    if ($product_id) {
      $product = Product::where('id', $product_id)
      ->where('status', true)->first();

      if ($product) {
        $pid = $product->id;
      }
    }

    if ($category_id) {
      if ($user_id) {
        $category = Category::where('id', $category_id)
          ->where('created_by', $user_id)->first();

        if ($category) {
          $cid = $category->id;
        }
      }
    }


    $rules = Tracking::$rules;
    $rules['ip'] = [
      'required',
      function ($attribute, $value, $fail) use ($pid, $user_id) {

        if ($pid) {
          $track = Tracking::where('ip', $value)
                ->where('product_id', $pid)
                ->where('created_by', $user_id)
                ->orderBy('created_at', 'desc')->first();
          if($track) {
            $settings = Setting::where('key', 'expiry_ip_tracking')->first();
            if($track->created_at > Carbon::now()->subDays((int) $settings->value)->format("Y-m-d H:i:s")) {
              return $fail('Duplicate ip');
            }
          }
        } else {
          return $fail('Missing product');
        }
      }
    ];
    $params = [
      "product_id" => $pid,
      "category_id" => $cid,
      "ip" => $ip,
      "referer" => $request->headers->get('referer'),
      "user_agent" => $request->headers->get('User-Agent')
    ];
    $validator = Validator::make($params, $rules);
    // $this->validate($request, $rules);
    if (!$validator->fails()) {
      // $ip = $request->ip();

      $settings = Setting::all();

      foreach($settings as $setting) {
        if ($setting->key == 'geo_ip_api_key') {
          $geo_ip_api = $setting->value;
        }
        if ($setting->key == 'country_allowed') {
          $country_allowed = json_decode($setting->value);
        }
      }

      // $link = "https://api.ipgeolocation.io/ipgeo";
      // $link .= "?apiKey=".$geo_ip_api."&ip=".$ip;
      $link = "https://api.ipregistry.co/".$ip."?key=".$geo_ip_api;

      $curl = curl_init($link);

      curl_setopt($curl, CURLOPT_URL, $link);
      curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
      // curl_setopt($http, CURLOPT_SSL_VERIFYHOST, 2);
      // curl_setopt($http, CURLOPT_SSL_VERIFYPEER, 2);
      $http_result = curl_exec($curl);
      $http_status = curl_getinfo($curl, CURLINFO_HTTP_CODE);
      curl_close($curl);

      if ($http_status == 200) {
        $result = json_decode($http_result);

        $params['city'] = $result->location->city;
        $params['state'] = $result->location->region->name;
        $params['country'] = $result->location->country->name;

        $params['created_by'] = $user_id;
        $params['updated_by'] = $user_id;

        $country_allowed = array_map( 'strtolower', $country_allowed );

        // echo $domains[0];

        if (in_array(strtolower($result->location->country->name), $country_allowed)) {
          $tracking = Tracking::create($params);
        }
      }

    }

    if ($product) {
      return redirect($product->url);
    }
    

    return "No available link";
  }

  public function show($id) {
    if (app('auth')->user()->hasRole('Superadmin') || app('auth')->user()->hasRole('Admin')) {
      $data = Tracking::findOrFail($id);
    } else {
      $data = Tracking::where('created_by', app('auth')->user()->id)
      ->findOrFail($id);
    }
    return response()->json($data);
  }

  public function update(Request $request, $id) {
    if (app('auth')->user()->hasRole('Superadmin') || app('auth')->user()->hasRole('Admin')) {
      $tracking = Tracking::findOrFail($id);
    } else {
      $tracking = Tracking::where('created_by', app('auth')->user()->id)
        ->findOrFail($id);
    }
    
    $rules = Tracking::$rules;

    $rules['tracking_no'] = [
      'required', 
      Rule::unique('trackings')->where(function ($query) use ($id) {
        return $query->where('id', '<>', $id);
      })
    ];

    $this->validate($request, $rules);
    $params = $request->all();

    $tracking->update($params);

    return response()->json($tracking, 200);
  }

  public function destroy($id) {
    $data = Tracking::findOrFail($id);

    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }

  public function location(Request $request) {
    $limit = $request->query('limit') ? (int) $request->query('limit') : 20;

    if (app('auth')->user()->hasRole('Superadmin')) {
      $trackings = Tracking::selectRaw("city, state, country, count(*) as count");
      
      if($request->query('user_id')) {
        $trackings = $trackings->where('created_by', $request->query('user_id'));
      }
      
      // ->orderBy('state','asc')
      // ->orderBy('city','asc');
    } else {
      $trackings = Tracking::selectRaw("city, state, country, count(*) as count")
      ->where('created_by', app('auth')->user()->id);
      // ->groupBy('city', 'state', 'country')
      // ->orderBy('count','desc');
      // ->orderBy('state','asc')
      // ->orderBy('city','asc');
    }

    if($request->query('from_date') && $request->query('end_date')) {
      $start = Carbon::createFromFormat('Y-m-d', $request->query('from_date'));
      $end = Carbon::createFromFormat('Y-m-d', $request->query('end_date'));
      $trackings = $trackings->whereBetween('created_at', [$start->startOfDay()->format('Y-m-d H:i:s'), $end->endOfDay()->format('Y-m-d H:i:s')]);
    }

    if($request->query('product_id')) {
      $trackings = $trackings->where('product_id', $request->query('product_id'));
    }

    if($request->query('category_id')) {
      $trackings = $trackings->where('created_by', $request->query('category_id'));
    }

    $trackings=$trackings->groupBy('city', 'state', 'country')
      ->orderBy('count','desc');

    return response()->json($trackings->paginate($limit));
  }

  public function weekly(Request $request) {
    $start = Carbon::now()->subDay(7);
    $end = Carbon::now();

    $arr = [];
    if (app('auth')->user()->hasRole('Superadmin') || app('auth')->user()->hasRole('Admin')) {

      while($start->lte($end)) {
        $day = $start->format('Y-m-d');
  
        $tracking = Tracking::where('created_at', 'like', $day.'%');

        if ($request->query('user_id')) {
          $tracking = $tracking->where('created_by', $request->query('user_id'));
        }

        if ($request->query('product_id')) {
          $tracking = $tracking->where('product_id', $request->query('product_id'));
        }

        $tracking = $tracking->count();
        $arr[$day] = [
          'total' => $tracking
        ];
        $start->addDay();
      }
    } else {
      while($start->lte($end)) {
        $day = $start->format('Y-m-d');
  
        $tracking = Tracking::where('created_at', 'like', $day.'%')
          ->where('created_by', app('auth')->user()->id);

        if ($request->query('product_id')) {
          $tracking = $tracking->where('product_id', $request->query('product_id'));
        }

        $tracking = $tracking->count();

        $arr[$day] = [
          'total' => $tracking
        ];
        $start->addDay();
      }
    }

    return response()->json($arr);
  }

  public function monthly(Request $request) {
    $start = Carbon::now()->startOfMonth();
    $end = Carbon::now()->endOfMonth();

    $arr = [];
    if (app('auth')->user()->hasRole('Superadmin') || app('auth')->user()->hasRole('Admin')) {

      while($start->lte($end)) {
        $day = $start->format('Y-m-d');
  
        $tracking = Tracking::where('created_at', 'like', $day.'%');

        if ($request->query('user_id')) {
          $tracking = $tracking->where('created_by', $request->query('user_id'));
        }

        if ($request->query('product_id')) {
          $tracking = $tracking->where('product_id', $request->query('product_id'));
        }

        $tracking = $tracking->count();
        $arr[$day] = [
          'total' => $tracking
        ];
        $start->addDay();
      }
    } else {
      while($start->lte($end)) {
        $day = $start->format('Y-m-d');
  
        $tracking = Tracking::where('created_at', 'like', $day.'%')
          ->where('created_by', app('auth')->user()->id);

        if ($request->query('product_id')) {
          $tracking = $tracking->where('product_id', $request->query('product_id'));
        }
        $tracking = $tracking->count();

        $arr[$day] = [
          'total' => $tracking
        ];
        $start->addDay();
      }
    }

    return response()->json($arr);
  }

  public function yearly(Request $request) {
    $start = Carbon::now()->startOfYear();
    $end = Carbon::now()->endOfYear();

    $arr = [];
    if (app('auth')->user()->hasRole('Superadmin') || app('auth')->user()->hasRole('Admin')) {

      while($start->lte($end)) {
        $day = $start->format('Y-m');
  
        $tracking = Tracking::where('created_at', 'like', $day.'%');

        if ($request->query('user_id')) {
          $tracking = $tracking->where('created_by', $request->query('user_id'));
        }

        if ($request->query('product_id')) {
          $tracking = $tracking->where('product_id', $request->query('product_id'));
        }

        $tracking = $tracking->count();
        $arr[$day] = [
          'total' => $tracking
        ];
        $start->addMonth();
      }
    } else {
      while($start->lte($end)) {
        $day = $start->format('Y-m');
  
        $tracking = Tracking::where('created_at', 'like', $day.'%')
          ->where('created_by', app('auth')->user()->id);

        if ($request->query('product_id')) {
          $tracking = $tracking->where('product_id', $request->query('product_id'));
        }

        $tracking = $tracking->count();

        $arr[$day] = [
          'total' => $tracking
        ];
        $start->addMonth();
      }
    }

    return response()->json($arr);
  }

  public function unknown_courier() {
    $tracking = Tracking::whereNull('courier_id')->count();

    return response()->json([
      'total' => $tracking
    ]);
  }

  public function dashboard(Request $request) {
    $end = Carbon::now()->endOfMonth();
    $start = Carbon::now()->startOfMonth();
    $end_prev = Carbon::now()->subMonth()->endOfMonth();
    $start_prev = Carbon::now()->subMonth()->startOfMonth();
    $start_day = Carbon::now()->startOfDay();
    $end_day = Carbon::now()->endOfDay();

    return response()->json([
      'total_today' => $this->get_dashboard_data($start_day, $end_day, $request),
      'total_this_month' => $this->get_dashboard_data($start, $end, $request),
      'total_last_month' => $this->get_dashboard_data($start_prev, $end_prev, $request)
    ]);
  }

  private function get_dashboard_data($start, $end, $request) {

    if (app('auth')->user()->hasRole('Superadmin') || app('auth')->user()->hasRole('Admin')) {
      $tracking = Tracking::whereBetween('created_at', [$start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s')]);
      if ($request->query('user_id')) {
        $tracking = $tracking->where('created_by', $request->query('user_id'));
      }
    } else {
      $tracking = Tracking::where('created_by', app('auth')->user()->id)
        ->whereBetween('created_at', [$start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s')]);
    }
    
    return $tracking->count();
  }
}