<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\DailySummarySales;
use Carbon\Carbon;

class DailySummarySalesController extends Controller
{

  public function index(Request $request) {
    if(app('auth')->user()->hasRole('Superadmin')) {
      $data = DailySummarySales::with(['user']);
    } else {
      $data = DailySummarySales::where('user_id', app('auth')->user()->id)
        ->with(['user']);
    }
    if($request->query('report_date')) {
      $data = $data->where('report_date', $request->query('report_date'));
    }
    if($request->query('from_date') && $request->query('end_date')) {
      $data = $data->whereBetween('report_date', [$request->query('from_date'), $request->query('end_date')]);
    }
    $data = $data->get();
    return response()->json($data);
  }

  public function store(Request $request) {
    $params = $request->all();
    $rules = DailySummarySales::$rules;

    $params['user_id'] = app('auth')->user()->hasRole('Superadmin') ? $params['user_id'] : app('auth')->user()->id;

    $rules['report_date'] = [
      'required', 
      Rule::unique('daily_summary_sales')->where(function ($query) use ($params) {
        return $query->where('user_id', $params['user_id'])
                ->where('report_date', $params['report_date']);
      })
    ];
    $this->validate($request, $rules);

    $data = DailySummarySales::create($params);
    return response()->json($data, 201);
  }

  public function show($id) {
    return response()->json(DailySummarySales::with(['user'])->findOrFail($id));
  }

  public function update($id, Request $request) {
    $data = DailySummarySales::findOrFail($id);

    if(!app('auth')->user()->hasRole('Superadmin')) {
      if ($data->user_id != app('auth')->user()->id) {
        return response()->json(['error' => 'Unauthorized - Cannot change other\'s data.'], 401);
      }
    } 
    $user_id = app('auth')->user()->hasRole('Superadmin') ? $data->user_id : app('auth')->user()->id;

    $rules = DailySummarySales::$rules;
    $rules['report_date'] = [
      'required', 
      Rule::unique('daily_summary_sales')->where(function ($query) use ($id, $data, $user_id) {
        return $query->where('id', '<>', $id)
                    ->where('user_id', $user_id)
                    ->where('report_date', $data->report_date);
      })
    ];
    $this->validate($request, $rules);
    $params = $request->all();
    $data->update($params);
    return response()->json($data, 200);
  }

  public function destroy($id) {
    $data = DailySummarySales::with(['user'])->findOrFail($id);

    if(!app('auth')->user()->hasRole('Superadmin')) {
      if($data->user_id != app('auth')->user()->id) {
        return response()->json(['message' => 'Unauthorized - Cannot delete other\'s data.'], 401);
      }
    }
    
    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }

  public function dashboard(Request $request) {
    $end = Carbon::now()->endOfMonth();
    $start = Carbon::now()->startOfMonth();
    $end_prev = Carbon::now()->subMonth()->endOfMonth();
    $start_prev = Carbon::now()->subMonth()->startOfMonth();

    $arr = $this->get_dashboard_data($start, $end);
    $arr_prev = $this->get_dashboard_data($start_prev, $end_prev);

    return response()->json([
      'total_leads_follow_up' => $arr['total_leads_follow_up'],
      'total_sales_repeat_order'  => $arr['total_sales_repeat_order'],
      'total_sales_unpaid'  => $arr['total_sales_unpaid'],
      'total_payment_collected' => $arr['total_payment_collected'],
      'commission'  => $arr['commission'],
      'total_leads_follow_up_prev' => $arr_prev['total_leads_follow_up'],
      'total_sales_repeat_order_prev'  => $arr_prev['total_sales_repeat_order'],
      'total_sales_unpaid_prev'  => $arr_prev['total_sales_unpaid'],
      'total_payment_collected_prev' => $arr_prev['total_payment_collected'],
      'commission_prev' => $arr_prev['commission']
    ]);
  }

  private function get_dashboard_data($start, $end) {
    //$end = Carbon::now()->endOfMonth();
    //$start = Carbon::now()->startOfMonth();
    $arr = [];
    $total_payment_collected = 0;
    $total_follow_up = 0;
    $total_sales_repeat = 0;
    $total_sales_unpaid = 0;

    while($start->lte($end)) {
      // info($last_week->format('D'));
      $sum_avg = 0;
      $day = $start->format('Y-m-d');

      $sales = DailySummarySales::where('report_date', $day)->with(['user']);

      if(app('auth')->user()->hasRole('Sales')) {
        $sales = $sales->where('user_id', app('auth')->user()->id);
      }

      $sales = $sales->get();

      if (count($sales) > 0) {
        foreach($sales as $sale) {
          $total_follow_up += $sale->total_follow_up;
          $total_sales_repeat += $sale->sales_repeat_order;
          $total_sales_unpaid += $sale->sales_unpaid;
          $total_payment_collected += $sale->total_payment_collected;
        }
      }
      
      $start->addDay();
    }
    $commission = $total_payment_collected * 0.02;

    return [
      'total_leads_follow_up' => $total_follow_up,
      'total_sales_repeat_order'  => $total_sales_repeat,
      'total_sales_unpaid'  => $total_sales_unpaid,
      'total_payment_collected' => $total_payment_collected,
      'commission'  => $commission
    ];
  }
}