<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\DailyReportMarketing;
use App\User;
use Carbon\Carbon;

class DailyReportMarketingController extends Controller
{

  public function index(Request $request) {
    if(app('auth')->user()->hasRole('Superadmin')) {
      $data = DailyReportMarketing::with(['product', 'user']);
    } else {
      $data = DailyReportMarketing::where('user_id', app('auth')->user()->id)
        ->with(['product', 'user']);
    }

    if($request->query('from_date') && $request->query('end_date')) {
      $data = $data->whereBetween('report_date', [$request->query('from_date'), $request->query('end_date')]);
    }
    $data = $data->get();
    foreach($data as $d) {
      $d->kpk = $d->total_daily_leads > 0 ? (float) $d->daily_ads_spending / $d->total_daily_leads : 0;
    }
    return response()->json($data);
  }

  public function store(Request $request) {
    $params = $request->all();
    $rules = DailyReportMarketing::$rules;

    $params['user_id'] = app('auth')->user()->hasRole('Superadmin') ? $params['user_id'] : app('auth')->user()->id;

    $rules['report_date'] = [
      'required', 
      Rule::unique('daily_report_marketing')->where(function ($query) use ($params) {
        return $query->where('product_id', $params['product_id'] )
                ->where('user_id', $params['user_id'])
                ->where('report_date', $params['report_date']);
      })
    ];
    $this->validate($request, $rules);
    // if (app('auth')->user()->hasRole('Marketing')) {
    //   $params['user_id'] = app('auth')->user()->id;
    // }

    $data = DailyReportMarketing::create($params);
    return response()->json($data, 201);
  }

  public function show($id) {
    return response()->json(DailyReportMarketing::with(['product', 'user'])->findOrFail($id));
  }

  public function update($id, Request $request) {
    $data = DailyReportMarketing::findOrFail($id);

    if(!app('auth')->user()->hasRole('Superadmin')) {
      if ($data->user_id != app('auth')->user()->id) {
        return response()->json(['error' => 'Unauthorized - Cannot change other\'s data.'], 401);
      }
    } 
    $user_id = app('auth')->user()->hasRole('Superadmin') ? $data->user_id : app('auth')->user()->id;

    $rules = DailyReportMarketing::$rules;
    $rules['report_date'] = [
      'required', 
      Rule::unique('daily_report_marketing')->where(function ($query) use ($id, $data, $user_id) {
        return $query->where('id', '<>', $id)
                    ->where('user_id', $user_id)
                    ->where('product_id', $data->product_id)
                    ->where('report_date', $data->report_date);
      })
    ];
    $this->validate($request, $rules);
    $params = $request->all();
    $data->update($params);
    return response()->json($data, 200);
  }

  public function destroy($id) {
    $data = DailyReportMarketing::with(['product', 'user'])->findOrFail($id);

    if(!app('auth')->user()->hasRole('Superadmin')) {
      if($data->user_id != app('auth')->user()->id) {
        return response()->json(['message' => 'Unauthorized - Cannot delete other\'s data.'], 401);
      }
    }
    
    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }

  public function dashboard(Request $request) {
    $end = Carbon::now()->endOfMonth();
    $start = Carbon::now()->startOfMonth();
    $end_prev = Carbon::now()->subMonth()->endOfMonth();
    $start_prev = Carbon::now()->subMonth()->startOfMonth();

    $arr = $this->get_dashboard_data($start, $end);
    $arr_prev = $this->get_dashboard_data($start_prev, $end_prev);

    $data_arr = [];
    foreach($arr['leads'] as $k => $v) {
      $start = Carbon::now()->startOfMonth();
      $end = Carbon::now();
      $day_arr = [
        'product' => $k
      ];

      while($start->lte($end)) {
        $day = $start->format('Y-m-d');
        if (array_key_exists($day, $v)) {
          $day_arr[$day] = $v[$day];
        } else {
          $day_arr[$day] = 0;
        }
        $start->addDay();
      }
      $data_arr[] = $day_arr;
    }
    return response()->json([
      'avg_kpk' => $arr['avg_kpk'],
      'avg_kpk_prev' => $arr_prev['avg_kpk'],
      'total_ads_spending'  => $arr['total_ads_spending'],
      'total_ads_spending_prev'  => $arr_prev['total_ads_spending'],
      'total_leads' => $arr['total_leads'],
      'total_leads_prev' => $arr_prev['total_leads'],
      'commission' => $arr['commission'],
      'commission_prev' => $arr_prev['commission'],
      'leads' => $data_arr
    ]);
  }

  public function summary(Request $request) {
    $date = date('Y-m-d');
    if ($request->query('date')) {
      $date = $request->query('date');
    }
    $users = User::role('Marketing')->orderBy('name', 'asc')->get();

    $data = [];
    foreach($users as $user) {
      $arr = [];
      $total_ads_spending = 0;
      $total_leads = 0;

      $costs = DailyReportMarketing::where('report_date', $date)
        ->where('user_id', $user->id)
        ->with(['product'])->get();
      
      foreach ($costs as $cost) {
        if(!in_array($cost->product->code, $arr)) {
          $arr[] = $cost->product->code;
        }
        $total_ads_spending += $cost->daily_ads_spending;
        $total_leads += $cost->total_daily_leads;
      }
      $data[] = [
        'date'  => $date,
        'name'  => $user->name,
        'total_ads_spending'  => $total_ads_spending,
        'total_leads' => $total_leads,
        'avg_kpk' => $total_leads > 0 ? $total_ads_spending/$total_leads : 0,
        'products'  => implode(', ', $arr)
      ];
    }

    return response()->json($data);
  }

  public function weekly(Request $request) {
    $start = Carbon::now()->startOfWeek(Carbon::MONDAY);
    $end = Carbon::now()->endOfWeek(Carbon::SUNDAY);

    $data = [];
    while($start->lte($end)) {
      $day = $start->format('Y-m-d');

      $sales = DailyReportMarketing::where('report_date', $day)->get();

      $total_ads_spending = 0;
      $total_leads = 0;
      foreach($sales as $sale) {
        $total_ads_spending += $sale->daily_ads_spending;
        $total_leads += $sale->total_daily_leads;
      }
      $data[$day] = [
        'total_ads_spending'  => (float) $total_ads_spending,
        'total_leads' => $total_leads
      ];
      $start->addDay();
    }

    return response()->json($data);
  }

  public function monthly(Request $request) {
    $end = Carbon::now()->endOfMonth();
    $start = Carbon::now()->startOfMonth();

    $data = [];
    while($start->lte($end)) {
      $day = $start->format('Y-m-d');

      $sales = DailyReportMarketing::where('report_date', $day)->get();

      $total_ads_spending = 0;
      $total_leads = 0;
      foreach($sales as $sale) {
        $total_ads_spending += $sale->daily_ads_spending;
        $total_leads += $sale->total_daily_leads;
      }
      $data[$day] = [
        'total_ads_spending'  => (float) $total_ads_spending,
        'total_leads' => $total_leads
      ];
      $start->addDay();
    }

    return response()->json($data);
  }

  public function yearly(Request $request) {
    $end = Carbon::now()->endOfYear();
    $start = Carbon::now()->startOfYear();

    $data = [];
    while($start->lte($end)) {
      $day = $start->format('Y-m');

      $sales = DailyReportMarketing::where('report_date', 'like', $day.'%')->get();

      $total_ads_spending = 0;
      $total_leads = 0;
      foreach($sales as $sale) {
        $total_ads_spending += $sale->daily_ads_spending;
        $total_leads += $sale->total_daily_leads;
      }
      $data[$day] = [
        'total_ads_spending'  => (float) $total_ads_spending,
        'total_leads' => $total_leads
      ];
      $start->addMonth();
    }

    return response()->json($data);
  }

  private function get_dashboard_data($start, $end) {
    //$end = Carbon::now()->endOfMonth();
    //$start = Carbon::now()->startOfMonth();
    $arr = [];
    $total_leads = 0;
    $total_ads_spending = 0;
    $commission = 0;
    $avg_kpk = 0;
    $total_day = 0;

    while($start->lte($end)) {
      // info($last_week->format('D'));
      $sum_avg = 0;
      $day = $start->format('Y-m-d');

      $costs = DailyReportMarketing::where('report_date', $day)->with(['product']);

      if(app('auth')->user()->hasRole('Marketing')) {
        $costs = $costs->where('user_id', app('auth')->user()->id);
      }

      $costs = $costs->get();

      if (count($costs) > 0) {
        foreach($costs as $cost) {
          $avg = $cost->total_daily_leads > 0 ? $cost->daily_ads_spending / $cost->total_daily_leads : 0;
          $total_ads_spending += $cost->daily_ads_spending;
          $total_leads += $cost->total_daily_leads;
          $sum_avg += $avg;

          if (array_key_exists($cost->product->name, $arr)) {
            $arr[$cost->product->name][$day] = $cost->total_daily_leads;
          } else {
            $arr[$cost->product->name] = [
              $day => $cost->total_daily_leads
            ];
          }
        }
  
        $avg_kpk += $sum_avg;
        $total_day++;
      }
      
      $start->addDay();
    }

    return [
      'avg_kpk' => $total_day > 0 ? (float) $avg_kpk / $total_day : 0,
      'total_ads_spending'  => (float) $total_ads_spending,
      'total_leads' => $total_leads,
      'commission' => (float) $total_leads * 0.5,
      'leads' => $arr
    ];
  }
}