<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\Sale;
use App\User;
use App\CampaignLead;
use App\CampaignCost;
use App\Product;
use App\WorkingDay;
use App\SalesTarget;
use App\Setting;
use Carbon\Carbon;

class SaleController extends Controller
{

  public function index(Request $request) {
    $data = Sale::with(['campaign', 'lead', 'product',  'assignee']);

    if($request->query('from_date') && $request->query('end_date')) {
      $data = $data->whereBetween('sales_date', [$request->query('from_date'), $request->query('end_date')]);
    }

    if(app('auth')->user()->hasRole('Sales Leader')) {
      $childs = User::where('parent_id', app('auth')->user()->id)->pluck('id')->toArray();
      $data = $data->whereIn('assigned_to', $childs);
    }

    if(app('auth')->user()->hasRole('Sales')) {
      $data = $data->where('assigned_to', app('auth')->user()->id);
    }

    if(app('auth')->user()->hasRole('Logistic')) {
      $data = $data->where('call_status', 'C')
                ->where('request_shipping', true);
    }
    
    $data = $data->orderBy('sales_date', 'desc')->get();
    return response()->json($data);
  }

  public function weekly(Request $request) {
    $now = Carbon::now();
    $last_week = Carbon::now()->subWeek();

    if(app('auth')->user()->hasRole('Sales Leader')) {
      $users = User::where('parent_id', app('auth')->user()->id)->pluck('id')->toArray();
      $working_days = WorkingDay::where('user_id', app('auth')->user()->id)
        ->where('is_working', true)->whereBetween('date', [$last_week->format('Y-m-d'), $now->format('Y-m-d')])
        ->pluck('date')->toArray();
      $sales_target = SalesTarget::where('user_id', app('auth')->user()->id)
        ->where('date', $now->format('Y-m'))->first();
    }
    if(app('auth')->user()->hasRole('Sales')) {
      $users = [app('auth')->user()->id];
      //$users = User::where('parent_id', app('auth')->user()->id)->pluck('id')->toArray();
      $working_days = WorkingDay::where('user_id', app('auth')->user()->parent_id)
        ->where('is_working', true)->whereBetween('date', [$last_week->format('Y-m-d'), $now->format('Y-m-d')])
        ->pluck('date')->toArray();
      $sales_target = SalesTarget::where('user_id', app('auth')->user()->parent_id)
        ->where('date', $now->format('Y-m'))->first();
    }
    if(app('auth')->user()->hasRole('Admin') || app('auth')->user()->hasRole('Marketing')) {
      $users = User::role('Sales')->pluck('id')->toArray();
    }
    // info(app('auth')->user());
    
    // info($last_week.' '.$now);
    // info($sales_target);
    $arr = [];
    while($last_week->lte($now)) {
      // info($last_week->format('D'));
      $day = $last_week->format('D');
      $expected = 0;
      
      $total_sales = Sale::whereIn('assigned_to', $users)->where('sales_date', $last_week->format('Y-m-d'))
        ->where('call_status', 'c')->sum('total_sales');

      if(app('auth')->user()->hasRole('Admin')) {
        $total_costs = CampaignCost::where('cost_date', $last_week->format('Y-m-d'))
        ->where('is_confirmed', true)->sum('amount');
        $arr[$day] = [
          'sales' => $total_sales ? (float) $total_sales : 0,
          'costs' => $total_costs ? (float) $total_costs : 0
        ];
      } else if(app('auth')->user()->hasRole('Marketing')) {
        $total_costs = CampaignCost::where('created_by', app('auth')->user()->id)->where('cost_date', $last_week->format('Y-m-d'))
        ->where('is_confirmed', true)->sum('amount');
        $arr[$day] = [
          'sales' => $total_sales ? (float) $total_sales : 0,
          'costs' => $total_costs ? (float) $total_costs : 0
        ];
      } else {
        if(in_array($last_week->format('Y-m-d'), $working_days)) {
          if($sales_target) {
            $expected = $sales_target->daily_target;
          }
        }
  
        
        info($last_week->format('D').' '.$expected.' '.$total_sales);
        $arr[$day] = [
          'actual'    => $total_sales ? (float) $total_sales : 0,
          'expected'  => $expected
        ];
      }
      
      $last_week->addDay();
    }
    return response()->json($arr);
  }

  public function monthly(Request $request) {
    $now = Carbon::now()->endOfMonth();
    $last_week = Carbon::now()->startOfMonth();

    if(app('auth')->user()->hasRole('Sales Leader')) {
      $users = User::where('parent_id', app('auth')->user()->id)->pluck('id')->toArray();
      $working_days = WorkingDay::where('user_id', app('auth')->user()->id)
        ->where('is_working', true)->whereBetween('date', [$last_week->format('Y-m-d'), $now->format('Y-m-d')])
        ->pluck('date')->toArray();
      $sales_target = SalesTarget::where('user_id', app('auth')->user()->id)
        ->where('date', $now->format('Y-m'))->first();
    }
    if(app('auth')->user()->hasRole('Sales')) {
      $users = [app('auth')->user()->id];
      //$users = User::where('parent_id', app('auth')->user()->id)->pluck('id')->toArray();
      $working_days = WorkingDay::where('user_id', app('auth')->user()->parent_id)
        ->where('is_working', true)->whereBetween('date', [$last_week->format('Y-m-d'), $now->format('Y-m-d')])
        ->pluck('date')->toArray();
      $sales_target = SalesTarget::where('user_id', app('auth')->user()->parent_id)
        ->where('date', $now->format('Y-m'))->first();
    }

    if(app('auth')->user()->hasRole('Admin') || app('auth')->user()->hasRole('Marketing')) {
      $users = User::role('Sales')->pluck('id')->toArray();
    }
    // info(app('auth')->user());
    
    // info($last_week.' '.$now);
    // info($sales_target);
    $arr = [];
    while($last_week->lte($now)) {
      // info($last_week->format('D'));
      $day = $last_week->format('Y-m-d');
      $expected = 0;
      
      $total_sales = Sale::whereIn('assigned_to', $users)->where('sales_date', $last_week->format('Y-m-d'))
        ->where('call_status', 'c')->sum('total_sales');

      if(app('auth')->user()->hasRole('Admin')) {
        $total_costs = CampaignCost::where('cost_date', $last_week->format('Y-m-d'))
        ->where('is_confirmed', true)->sum('amount');
        $arr[$day] = [
          'sales' => $total_sales ? (float) $total_sales : 0,
          'costs' => $total_costs ? (float) $total_costs : 0
        ];
      } else if(app('auth')->user()->hasRole('Marketing')) {
        $total_costs = CampaignCost::where('created_by', app('auth')->user()->id)->where('cost_date', $last_week->format('Y-m-d'))
        ->where('is_confirmed', true)->sum('amount');
        $arr[$day] = [
          'sales' => $total_sales ? (float) $total_sales : 0,
          'costs' => $total_costs ? (float) $total_costs : 0
        ];
      } else {
        if(in_array($last_week->format('Y-m-d'), $working_days)) {
          if($sales_target) {
            $expected = $sales_target->daily_target;
          }
        }
  
        
        info($last_week->format('Y-m-d').' '.$expected.' '.$total_sales);
        $arr[$day] = [
          'actual'    => $total_sales ? (float) $total_sales : 0,
          'expected'  => $expected
        ];
      }
      $last_week->addDay();
    }
    return response()->json($arr);
  }

  public function yearly(Request $request) {
    $now = Carbon::now()->endOfYear();
    $last_week = Carbon::now()->startOfYear();

    if(app('auth')->user()->hasRole('Sales Leader')) {
      $users = User::where('parent_id', app('auth')->user()->id)->pluck('id')->toArray();
      $working_days = WorkingDay::where('user_id', app('auth')->user()->id)
        ->where('is_working', true)->whereBetween('date', [$last_week->format('Y-m-d'), $now->format('Y-m-d')])
        ->pluck('date')->toArray();
      $sales_target = SalesTarget::where('user_id', app('auth')->user()->id)
        ->where('date', $now->format('Y-m'))->first();
    }
    if(app('auth')->user()->hasRole('Sales')) {
      $users = [app('auth')->user()->id];
      //$users = User::where('parent_id', app('auth')->user()->id)->pluck('id')->toArray();
      $working_days = WorkingDay::where('user_id', app('auth')->user()->parent_id)
        ->where('is_working', true)->whereBetween('date', [$last_week->format('Y-m-d'), $now->format('Y-m-d')])
        ->pluck('date')->toArray();
      $sales_target = SalesTarget::where('user_id', app('auth')->user()->parent_id)
        ->where('date', $now->format('Y-m'))->first();
    }
    if(app('auth')->user()->hasRole('Admin') || app('auth')->user()->hasRole('Marketing')) {
      $users = User::role('Sales')->pluck('id')->toArray();
    }
    // info(app('auth')->user());
    
    // info($last_week.' '.$now);
    // info($sales_target);
    $arr = [];
    while($last_week->lte($now)) {
      // info($last_week->format('D'));
      $day = $last_week->format('M');
      $expected = 0;

      $total_sales = Sale::whereIn('assigned_to', $users)->where('sales_date', 'like', $last_week->format('Y-m').'%')
          ->where('call_status', 'c')->sum('total_sales');

      if(app('auth')->user()->hasRole('Admin')) {
        $total_costs = CampaignCost::where('cost_date', 'like', $last_week->format('Y-m').'%')
        ->where('is_confirmed', true)->sum('amount');
        $arr[$day] = [
          'sales' => $total_sales ? (float) $total_sales : 0,
          'costs' => $total_costs ? (float) $total_costs : 0
        ];
      } else if(app('auth')->user()->hasRole('Marketing')) {
        $total_costs = CampaignCost::where('created_by', app('auth')->user()->id)->where('cost_date', 'like', $last_week->format('Y-m').'%')
        ->where('is_confirmed', true)->sum('amount');
        $arr[$day] = [
          'sales' => $total_sales ? (float) $total_sales : 0,
          'costs' => $total_costs ? (float) $total_costs : 0
        ];
      } else {
        if(in_array($last_week->format('Y-m-d'), $working_days)) {
          if($sales_target) {
            $expected = $sales_target->daily_target;
          }
        }
  
        info($last_week->format('D').' '.$expected.' '.$total_sales);
        $arr[$day] = [
          'actual'    => $total_sales ? (float) $total_sales : 0,
          'expected'  => $expected
        ];
      }
      
      $last_week->addMonth();
    }
    return response()->json($arr);
  }

  public function summary(Request $request) {
    if(app('auth')->user()->hasRole('Sales Leader')) {
      $data = [];
      $agents = User::where('parent_id', app('auth')->user()->id)->orderBy('name', 'asc')->get();
      // info($sales_leaders);
      foreach($agents as $agent) {
        info($agent);
        $leads = CampaignLead::where('assigned_to', $agent->id);
        $sales = Sale::where('assigned_to', $agent->id);
        if($request->query('from_date') && $request->query('end_date')) {
          $leads = $leads->whereBetween('assigned_date',[$request->query('from_date'), $request->query('end_date')]);
          $sales = $sales->whereBetween('sales_date',[$request->query('from_date'), $request->query('end_date')]);
        }
        $leads = $leads->get();
        $sales = $sales->get();
        $total_close = 0;
        $total_qty = 0;
        $total_close_amount = 0;
        foreach($sales as $sale) {
          if ($sale->call_status == 'c') {
            $total_close++;
            $total_qty += $sale->qty;
            $total_close_amount += $sale->total_sales;
          }
        }
        $agent->total_leads = count($leads);
        $agent->total_sales = count($sales);
        $agent->total_close = $total_close;
        $agent->total_qty = $total_qty;
        $agent->total_close_amount = $total_close_amount;
        $data[] = $agent;
      }
    }

    if(app('auth')->user()->hasRole('Admin')) {
      $data = [];
      $sales_leaders = User::role('Sales Leader')->orderBy('name', 'asc')->get();
      // info($sales_leaders);
      foreach($sales_leaders as $sl) {
        $agents = User::where('parent_id', $sl->id)->pluck('id')->toArray();
        $leads = CampaignLead::whereIn('assigned_to', $agents);
        $sales = Sale::whereIn('assigned_to', $agents);
        if($request->query('from_date') && $request->query('end_date')) {
          $leads = $leads->whereBetween('assigned_date',[$request->query('from_date'), $request->query('end_date')]);
          $sales = $sales->whereBetween('sales_date',[$request->query('from_date'), $request->query('end_date')]);
        }
        $leads = $leads->get();
        $sales = $sales->get();
        $total_close = 0;
        $total_qty = 0;
        $total_close_amount = 0;
        foreach($sales as $sale) {
          if ($sale->call_status == 'c') {
            $total_close++;
            $total_qty += $sale->qty;
            $total_close_amount += $sale->total_sales;
          }
        }
        $sl->total_agents = count($agents);
        $sl->total_leads = count($leads);
        $sl->total_sales = count($sales);
        $sl->total_close = $total_close;
        $sl->total_qty = $total_qty;
        $sl->total_close_amount = $total_close_amount;
        $data[] = $sl;
      }
    }
    return response()->json($data);
  }

  public function sum_monthly(Request $request) {
    $now = Carbon::now();
    $curr_month = Carbon::now()->format('Y-m');

    if(app('auth')->user()->hasRole('Sales Leader')) {
      $users = User::where('parent_id', app('auth')->user()->id)->pluck('id')->toArray();
      $total_sales = Sale::whereIn('assigned_to', $users)->where('sales_date', 'like', $curr_month.'%')
        ->where('call_status', 'c')->sum('total_sales');
    }
    if(app('auth')->user()->hasRole('Sales')) {
      $users = [app('auth')->user()->id];
      $total_sales = Sale::whereIn('assigned_to', $users)->where('sales_date', 'like', $curr_month.'%')
        ->where('call_status', 'c')->sum('total_sales');
    }
    
    return response()->json([
      'total_sales' => $total_sales ? (float) $total_sales : 0
    ]);
  }

  public function sum_daily(Request $request) {
    $now = Carbon::now();
    $curr_day = Carbon::now()->format('Y-m-d');

    if(app('auth')->user()->hasRole('Sales Leader')) {
      $users = User::where('parent_id', app('auth')->user()->id)->pluck('id')->toArray();
      $total_sales = Sale::whereIn('assigned_to', $users)->where('sales_date', 'like', $curr_day.'%')
        ->where('call_status', 'c')->sum('total_sales');
    }
    if(app('auth')->user()->hasRole('Sales')) {
      $users = [app('auth')->user()->id];
      $total_sales = Sale::whereIn('assigned_to', $users)->where('sales_date', 'like', $curr_day.'%')
        ->where('call_status', 'c')->sum('total_sales');
    }
    
    return response()->json([
      'total_sales' => $total_sales ? (float) $total_sales : 0
    ]);
  }

  public function store(Request $request) {
    $rules = Sale::$rules;
    
    $this->validate($request, $rules);
    $params = $request->all();

    if(app('auth')->user()->hasRole('Admin')) {
      $lead = CampaignLead::findOrFail($params['lead_id']);
    }
    if(app('auth')->user()->hasRole('Sales')) {
      $lead = CampaignLead::where('assigned_to', app('auth')->user()->id)
        ->where('id', $params['lead_id'])->firstOrFail();
    }
    $product = Product::find($lead->product_id);
    $params['campaign_id'] = $lead->campaign_id;
    $params['product_id'] = $lead->product_id; 
    $params['assigned_to'] = $lead->assigned_to ? $lead->assigned_to : app('auth')->user()->id;
    $params['price'] = $product->price;

    $settings = Setting::all();

    $sales_commission = 0;
    $marketing_commission = 0;

    foreach($settings as $setting) {
      if ($setting->key == 'sales_commission') {
        $sales_commission = (float) $setting->value;
      }
      if ($setting->key == 'marketing_commission') {
        $marketing_commission = (float) $setting->value;
      }
    }

    $params['commission_sales_percent'] = $sales_commission;
    $params['commission_marketing_percent'] = $marketing_commission;

    $params['commission_sales'] = $params['total_sales'] * ($sales_commission/100);
    $params['commission_marketing'] = $params['total_sales'] * ($marketing_commission/100);

    $data = Sale::updateOrCreate([
      'lead_id' => $params['lead_id']
    ], $params);
    return response()->json($data, 201);
  }

  public function show($id) {
    if(app('auth')->user()->hasRole('Admin')) {
      $data = Sale::with(['campaign', 'lead', 'product',  'assignee'])->findOrFail($id);
    }

    if(app('auth')->user()->hasRole('Sales Leader')) {
      $childs = User::where('parent_id', app('auth')->user()->id)->pluck('id')->toArray();
      $data = Sale::with(['campaign', 'lead', 'product',  'assignee'])->whereIn('assigned_to', $childs)
        ->where('id', $id)->firstOrFail();
    }

    if(app('auth')->user()->hasRole('Sales')) {
      $data = Sale::with(['campaign', 'lead', 'product',  'assignee'])->where('assigned_to', app('auth')->user()->id)
        ->where('id', $id)->firstOrFail();
    }
    return response()->json($data);
  }

  public function update(Request $request, $id) {
    if(app('auth')->user()->hasRole('Admin')) {
      $data = Sale::findOrFail($id);
    }
    if(app('auth')->user()->hasRole('Sales')) {
      $data = Sale::where('assigned_to', app('auth')->user()->id)
        ->where('id', $id)->firstOrFail();
    }

    $rules = Sale::$rules;
    
    $this->validate($request, $rules);
    $params = $request->all();

    if ($params['total_sales']) {
      $settings = Setting::all();

      $sales_commission = 0;
      $marketing_commission = 0;
      
      foreach($settings as $setting) {
        if ($setting->key == 'sales_commission') {
          $sales_commission = (float) $setting->value;
        }
        if ($setting->key == 'marketing_commission') {
          $marketing_commission = (float) $setting->value;
        }
      }

      $params['commission_sales_percent'] = $sales_commission;
      $params['commission_marketing_percent'] = $marketing_commission;

      $params['commission_sales'] = $params['total_sales'] * ($sales_commission/100);
      $params['commission_marketing'] = $params['total_sales'] * ($marketing_commission/100);
    }

    $data->update($params);
    return response()->json($data, 200);
  }

  public function destroy($id) {
    if(app('auth')->user()->hasRole('Admin')) {
      $data = Sale::findOrFail($id);
    }
    if(app('auth')->user()->hasRole('Sales')) {
      $data = Sale::where('assigned_to', app('auth')->user()->id)
        ->where('id', $id)->firstOrFail();
    }
    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }
}