<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\Referal;
use App\Tracking;
use App\Product;
use App\Category;
use App\Setting;
use App\User;
use Carbon\Carbon;

class ReferalController extends Controller
{

  public function index(Request $request) {
    $limit = $request->query('limit') ? (int) $request->query('limit') : 20;

    if (app('auth')->user()->hasRole('Superadmin')) {
      $data = Referal::with(['creator', 'category', 'product']); 

      if($request->query('user_id')) {
        $data = $data->where('created_by', $request->query('user_id'));
      }

    } else {
      $data = Referal::with(['category', 'product'])->where('created_by', app('auth')->user()->id); 
      
    }

    if($request->query('product_id')) {
      $data = $data->where('product_id', $request->query('product_id'));
    }

    if($request->query('category_id')) {
      $data = $data->where('category_id', $request->query('category_id'));
    }
    
    $data = $data->orderBy('created_at', 'desc')->paginate($limit);

    $setting = Setting::where('key', 'domain')->first();
    $domains = json_decode($setting->value);

    foreach ($data as $d) {
      $urls = [];
      foreach($domains as $domain) {
        $urls[] = $domain."/".$d->code;
      }
      $trackings = Tracking::where('referal_id',$d->id);

      if($request->query('from_date') && $request->query('end_date')) {
        $start = Carbon::createFromFormat('Y-m-d', $request->query('from_date'));
        $end = Carbon::createFromFormat('Y-m-d', $request->query('end_date'));
        $trackings = $trackings->whereBetween('created_at', [$start->startOfDay()->format('Y-m-d H:i:s'), $end->endOfDay()->format('Y-m-d H:i:s')]);
      }
      $trackings = $trackings->get();
      $commission = 0;
      foreach($trackings as $t) {
        $commission += $t->commission;
      }
      $d['tracking_count'] = count($trackings);
      $d['total_commission'] = number_format($commission, 2);
      $d['url'] = $urls;
    }
    return response()->json($data);
  }

  public function store(Request $request) {
    $params = $request->all();
    $user_id = app('auth')->user()->id;

    $rules = Referal::$rules;

    $rules['product_id'] = [
      'required', 
      function ($attribute, $value, $fail) use ($params, $user_id) {

        if ($params['product_id']) {
          $referal = Referal::where('product_id', $params['product_id'])
                ->where('created_by', $user_id);
          if ($params['category_id']) {
            $referal = $referal->where('category_id', $params['category_id']);
          }
          $referal = $referal->first();
          if($referal) {
            return $fail('Duplicate product/category');
          }
        } else {
          return $fail('Missing product');
        }
      }
    ];
    $params['created_by'] = $user_id;
    $this->validate($request, $rules);

    $referal = Referal::create($params);
    
    $data = Referal::findOrFail($referal->id);
    return response()->json($data, 201);
  }

  public function show($id) {
    if (app('auth')->user()->hasRole('Superadmin') || app('auth')->user()->hasRole('Admin')) {
      $data = Referal::with(['creator', 'category', 'product'])->findOrFail($id);
    } else {
      $data = Referal::with(['creator', 'category', 'product'])->where('created_by', app('auth')->user()->id)
      ->findOrFail($id);
    }
    return response()->json($data);
  }

  public function update(Request $request, $id) {
    if (app('auth')->user()->hasRole('Superadmin') || app('auth')->user()->hasRole('Admin')) {
      $data = Referal::findOrFail($id);
    } else {
      $data = Referal::where('created_by', app('auth')->user()->id)
        ->findOrFail($id);
    }
    $params = $request->all();
    
    $rules = Referal::$rules;

    $rules['product_id'] = [
      'required', 
      function ($attribute, $value, $fail) use ($id, $params, $user_id) {

        if ($params['product_id']) {
          $referal = Referal::where('product_id', $params['product_id'])
                ->where('created_by', $user_id)
                ->where('id', '<>', $id);
          if ($params['category_id']) {
            $referal = $referal->where('category_id', $params['category_id']);
          }
          $referal = $referal->first();
          if($referal) {
            return $fail('Duplicate product/category');
          }
        } else {
          return $fail('Missing product');
        }
      }
    ];
    $params['updated_by'] = $user_id;

    $this->validate($request, $rules);

    $data->update($params);

    return response()->json($data, 200);
  }

  public function destroy($id) {
    $data = Referal::findOrFail($id);

    $trackings = Tracking::where('referal_id', $id)->first();

    if ($trackings) {
      return response()->json(['message' => 'Unable to delete - Tracking data linked to this referal.'], 403);
    }

    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }

}