<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Order;
use App\Product;
use App\UserGroupProduct;

class OrderController extends Controller
{

  public function index(Request $request) {

    if(app('auth')->user()->hasRole('User')) {
      $data = Order::where('created_by', app('auth')->user()->id)
        ->with(['products', 'creator', 'order_groups']);
    } else {
      $data = Order::with(['products', 'creator', 'order_groups']);      
    }

    if($request->query('from_date') && $request->query('end_date')) {
      $data = $data->whereBetween('order_date', [$request->query('from_date'), $request->query('end_date')]);
    }

    if($request->query('from_ship_date') && $request->query('end_ship_date')) {
      $data = $data->whereBetween('shipping_date', [$request->query('shipping_date'), $request->query('shipping_date')]);
    }

    if($request->query('is_paid')) {
      $is_paid = filter_var($request->query('is_paid'), FILTER_VALIDATE_BOOLEAN);
      $data = $data->where('is_paid', $is_paid);
    }
    if($request->query('is_shipped')) {
      $is_shipped = filter_var($request->query('is_shipped'), FILTER_VALIDATE_BOOLEAN);
      $data = $data->where('is_shipped', $is_shipped);
    }
    return response()->json($data->get());
  }

  public function store(Request $request) {
    $this->validate($request, Order::$rules);

    $params = $request->all();
    $params['created_by'] = app('auth')->user()->id;
    $params['updated_by'] = app('auth')->user()->id;

    $order = Order::create($params);

    $usergroup_id = app('auth')->user()->usergroup_id;

    $products = UserGroupProduct::where('usergroup_id', $usergroup_id)->pluck('price', 'product_id')->toArray();

    $total = 0;
    if (isset($params['products']) && is_array($params['products'])) {
      foreach($params['products'] as $product) {
        if (array_key_exists($product['id'], $products)) {
          $total += $product['qty'] * $products[$product['id']];
          $order->products()->syncWithoutDetaching(array(
              $product['id'] => array(
                "qty" => $product['qty'],
                "cost_price" => $products[$product['id']],
                "total_cost_price" => $product['qty'] * $products[$product['id']],
                "created_by" => $params['created_by'],
                "updated_by" => $params['updated_by']
              )
            )
          );
        }
      }
    }

    $order->update([
      'total_cost_price' => $total
    ]);
    
    $data = Order::with(['products'])->findOrFail($order->id);
    return response()->json($data, 201);
  }

  public function show($id) {
    if (app('auth')->user()->hasRole('Superadmin')) {
      $data = Order::with(['products'])
      ->findOrFail($id);
    } else {
      $data = Order::where('created_by', app('auth')->user()->id)->with(['products'])
      ->findOrFail($id);
    }
    return response()->json($data);
  }

  public function update(Request $request, $id) {
    $order = Order::with(['products'])->findOrFail($id);
    $rules = Order::$rules;

    $this->validate($request, $rules);
    $params = $request->all();

    if ($order->is_paid && app('auth')->user()->hasRole('User')) {
      return response()->json(['error' => 'Cannot update details on paid order.'], 401);
    }

    $usergroup_id = app('auth')->user()->usergroup_id;

    $products = UserGroupProduct::where('usergroup_id', $usergroup_id)->pluck('price', 'product_id')->toArray();

    $total = 0;
    if (isset($params['products']) && is_array($params['products'])) {
      // $order->products()->delete();
      foreach($params['products'] as $product) {
        if (array_key_exists($product['id'], $products)) {
          $total += $product['qty'] * $products[$product['id']];
          $order->products()->syncWithoutDetaching(array(
            $product['id'] => array(
              "qty" => $product['qty'],
              "cost_price" => $products[$product['id']],
              "total_cost_price" => $product['qty'] * $products[$product['id']],
              "created_by" => $order->created_by ? $order->created_by : app('auth')->user()->id,
              "updated_by" => app('auth')->user()->id
            )
          )
        );
        }  
      }
      $params['total_cost_price'] = $total;
    }

    $order->update($params);

    return response()->json($order, 200);
  }

  public function destroy($id) {
    if (app('auth')->user()->hasRole('Superadmin')) {
      $data = Order::with(['products'])
      ->findOrFail($id);
    } else {
      $data = Order::where('created_by', app('auth')->user()->id)->with(['products'])
      ->findOrFail($id);
    }

    if ($data->is_paid || $data->is_shipped) {
      return response()->json(['error' => 'Cannot delete paid/shipped order.'], 401);
    }
    $data->products()->detach();
    $data->groups()->detach();
    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }

  public function bulkDestroy(Request $request) {
    $params = $request->all();
    if (is_array($params['orders'])) {
      if (app('auth')->user()->hasRole('Superadmin')) {
        $data = Order::where('is_paid', false)->whereIn('id', $params['orders'])->get();
      } else {
        $data = Order::where('is_paid', false)
        ->where('created_by', app('auth')->user()->id)
        ->whereIn('id', $params['orders'])->get();
      }

      foreach($data as $d) {
        $d->products()->detach();
        $d->groups()->detach();
        $d->delete();
      }

    }
    
    return response()->json(['message' => 'Delete Successfully'], 200);
  }

  public function upload(Request $request) {
    $params = $request->all();

    if (is_array($params['orders'])) {
      $products = Product::where('status', true)->pluck('id','sku')->toArray();
      $count_err = 0;

      $usergroup_id = app('auth')->user()->usergroup_id;
      $product_prices = UserGroupProduct::where('usergroup_id', $usergroup_id)->pluck('price', 'product_id')->toArray();


      foreach ($params['orders'] as $order) {
        $order['created_by'] = app('auth')->user()->id;
        $order['updated_by'] = app('auth')->user()->id;

        $total = 0;

        if (isset($order['products']) && is_array($order['products'])) {
          
          $sku_error = false;
          $new_products = [];
          foreach ($order['products'] as $product) {
            if (array_key_exists($product['sku'], $products)) {
              $new_products[$products[$product['sku']]] = $product;
            } else {
              $sku_error = true;
            }
          }

          if (!$sku_error) {
            $data = Order::create($order);
            foreach ($new_products as $key => $val) {
              $total += $val['qty'] * $product_prices[$key];
              $data->products()->syncWithoutDetaching(array(
                $key => array(
                    "qty" => $val['qty'],
                    "cost_price" => $product_prices[$key],
                    "total_cost_price" => $val['qty'] * $product_prices[$key],
                    "created_by" => $order['created_by'],
                    "updated_by" => $order['updated_by']
                  )
                )
              );
            }
            $data->update([
              'total_cost_price' => $total
            ]);
          } else {
            $count_err++;
          }
        }
        
      }
    }
    return response()->json(['message' => count($params['orders']) - $count_err . "/".count($params['orders']) .' orders uploaded successfully'], 200);
  }
}