<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\DeliveryOrder;
use App\Tracking;
use App\Courier;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade as PDF;

class DeliveryOrderController extends Controller
{

  public function index(Request $request) {

    $data = DeliveryOrder::with(['trackings', 'courier']) ;

    if($request->query('from_date') && $request->query('end_date')) {
      $start = Carbon::createFromFormat('Y-m-d', $request->query('from_date'));
      $end = Carbon::createFromFormat('Y-m-d', $request->query('end_date'));
      $data = $data->whereBetween('delivery_order_date', [$start->startOfDay()->format('Y-m-d H:i:s'), $end->endOfDay()->format('Y-m-d H:i:s')]);
    }
    
    return response()->json($data->get());
  }

  public function store(Request $request) {
    // $this->validate($request, Order::$rules);

    $params = $request->all();

    $rules = DeliveryOrder::$rules;
    $this->validate($request, $rules);

    $courier = Courier::findOrFail($params['courier_id']);

    $params['delivery_order_no'] = $courier->code.'-'.Carbon::now()->format("YmdHis");
    $params['delivery_order_date'] = Carbon::now()->format("Y-m-d H:i:s");
    $params['courier_id'] = $params['courier_id'];
    $params['created_by'] = app('auth')->user()->id;
    $params['updated_by'] = app('auth')->user()->id;

    $do = null;
    if (isset($params['tracking_id']) && is_array($params['tracking_id']) && count($params['tracking_id']) > 0) {

      $trackings = Tracking::whereIn('id', $params['tracking_id'])
        ->where('courier_id', $params['courier_id'])
        ->where('delivery_order_id', null)->get();

      if (count($trackings) > 0 ) {
        $do = DeliveryOrder::create($params);
        foreach($trackings as $tracking) {
          $tracking->update([
            'delivery_order_id' => $do->id
          ]);
        }
      }

    }
    if ($do) {
      $data = DeliveryOrder::with(['trackings'])->findOrFail($do->id);
      return response()->json($data, 201);
    } else {
      return response()->json(['error' => 'Tracking no has already been assigned to delivery orders.'], 404);
    }
    
  }

  public function show($id) {
    if (app('auth')->user()->hasRole('Superadmin')) {
      $data = DeliveryOrder::with(['trackings'])
      ->findOrFail($id);
    } else {
      $data = DeliveryOrder::where('created_by', app('auth')->user()->id)->with(['trackings'])
      ->findOrFail($id);
    }
    return response()->json($data);
  }

  public function update(Request $request, $id) {
    $order = DeliveryOrder::findOrFail($id);
    $rules = DeliveryOrder::$rules;

    $this->validate($request, $rules);
    $params = $request->all();

    $order->update($params);
    return response()->json($order, 200);
  }

  public function destroy($id) {
    if (app('auth')->user()->hasRole('Superadmin') || app('auth')->user()->hasRole('Admin')) {
      $data = DeliveryOrder::with(['trackings'])
      ->findOrFail($id);
    } else {
      $data = DeliveryOrder::where('created_by', app('auth')->user()->id)->with(['trackings'])
      ->findOrFail($id);
    }

    Tracking::where('delivery_order_id', $id)
      ->update([
        'delivery_order_id' => null
      ]);
    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }

  public function download($id) {
     
    $do = DeliveryOrder::with(['trackings', 'courier'])->findOrFail($id);
    
    // $parcels = [];
    // for($i=0; $i < 550; $i++) {
    //   $parcels[]['tracking_no'] = 100000 + $i;
    // }
    if ($do->trackings()->count() > 102) {
      $parcels = array_chunk($do->trackings->toArray(), 102);

      if ($do->trackings()->count() % 102 > 75) {
        $arr = array_chunk($parcels[count($parcels)-1], 75);
        array_pop($parcels);
        $parcels = array_merge($parcels, $arr);
      }
    } else {
      $parcels = array_chunk($do->trackings->toArray(), 75);
    }
    
    $data = [
      'do_no' => $do->delivery_order_no,
      'do_date' => Carbon::createFromFormat('Y-m-d H:i:s', $do->delivery_order_date)->format("d/m/Y"),
      'courier' => strtoupper($do->courier->name),
      'total' => $do->trackings()->count(),
      // 'parcels' => $do->trackings->toArray()
      // 'parcels' => [
      //     [
      //       ['tracking_no' =>11111],
      //       ['tracking_no' =>11112],
      //       ['tracking_no' =>11113],
      //       ['tracking_no' =>11114],
      //       ['tracking_no' =>11115]
      //     ],
      //   [
      //     ['tracking_no' =>11116],
      //     ['tracking_no' =>11117],
      //     ['tracking_no' =>11118],
      //     ['tracking_no' =>11119],
      //     ['tracking_no' =>111110]
      //   ]
      // ]
      'parcels' => $parcels
    ];
    $pdf = PDF::loadView('do', $data); 
    // $pdf->output();
    // $text = "Page {PAGE_NUM} of {PAGE_COUNT}";
    // $size = 8;
    $dom_pdf = $pdf->getDomPDF();
    $dom_pdf->set_option("enable_php", true);
    // $dom_pdf->setPaper('A4'); 
    // $font = $dom_pdf->getFontMetrics()->get_font("Verdana");
    // $width = $dom_pdf->getFontMetrics()->get_text_width($text, $font, $size) / 2;
    // $x = ($dom_pdf->get_width() - $width) / 2;
    // $y = $dom_pdf->get_height() - 35;
    // $pdf->getDomPDF()->getCanvas()->page_text($x, $y, $text, $font, $size); 
    return $pdf->stream($do->delivery_order_no.".pdf");
    // return view('do', $data);
  }

  public function daily(Request $request) {
    $params = $request->all();
    if(isset($params['from_date']) && isset($params['end_date']) && isset($params['courier_id'])) {
      $start = Carbon::parse($params['from_date'])->startOfDay();
      $end = Carbon::parse($params['end_date'])->endOfDay();

      $courier = Courier::findOrFail($params['courier_id']);

      $trackings = Tracking::where('delivery_order_id', null)
        ->where('courier_id', $params['courier_id'])
        ->whereBetween('scan_date', [$start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s')])->get();

      $do = null;
      if (count($trackings) > 0) {
        $do = DeliveryOrder::create([
          'delivery_order_no' => $courier->code.'-'.Carbon::now()->format("YmdHis"),
          'delivery_order_date' => Carbon::now()->format("Y-m-d H:i:s"),
          'courier_id' => $params['courier_id'],
          'created_by' => app('auth')->user()->id,
          'updated_by' => app('auth')->user()->id
        ]);
  
        foreach ($trackings as $tracking) {
          $tracking->update([
            'delivery_order_id' => $do->id
          ]);
        }
      }

      if ($do) {
        $data = DeliveryOrder::with(['trackings'])->findOrFail($do->id);
        return response()->json($data, 201);
      } else {
        return response()->json(['error' => 'Tracking no has already been assigned to delivery orders.'], 404);
      }
    }
    return response()->json(['error' => 'Missing tracking date from and date end.'], 403);
  }
}