<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\DailySales;
use App\DailyReportMarketing;
use App\DailyReportParcel;
use App\DailyReportSales;
use App\ParcelReturn;
use App\SalesFighter;
use Carbon\Carbon;

class DailySalesController extends Controller
{

  public function index(Request $request) {
    $data = DailySales::orderBy('report_date', 'desc');

    if($request->query('from_date') && $request->query('end_date')) {
      $data = $data->whereBetween('report_date', [$request->query('from_date'), $request->query('end_date')]);
    }
    // if($request->query('from_date_return') && $request->query('end_date_return')) {
    //   $data = $data->whereBetween('return_date', [$request->query('from_date_return'), $request->query('end_date_return')]);
    // }
    $data = $data->get();
    return response()->json($data);
  }

  public function store(Request $request) {
    $params = $request->all();
    $rules = DailySales::$rules;

    $this->validate($request, $rules);
    // if (app('auth')->user()->hasRole('Marketing')) {
    //   $params['user_id'] = app('auth')->user()->id;
    // }

    $data = DailySales::create($params);
    return response()->json($data, 201);
  }

  public function show($id) {
    return response()->json(DailySales::findOrFail($id));
  }

  public function update($id, Request $request) {
    $data = DailySales::findOrFail($id);
    $params = $request->all();

    $rules = DailySales::$rules;

    $rules['report_date'] = [
      'required', 
      Rule::unique('daily_sales_record')->where(function ($query) use ($id, $params) {
        return $query->where('id', '<>', $id)
                    ->where('report_date', $params['report_date']);
      })
    ];
    
    $this->validate($request, $rules);
    $params = $request->all();
    $data->update($params);
    return response()->json($data, 200);
  }

  public function destroy($id) {
    $data = DailySales::findOrFail($id);
    
    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }

  public function dashboard(Request $request) {
    $this_month = date('Y-m');
    $last_month = Carbon::now()->subMonth()->format('Y-m');

    $data = $this->get_dashboard_data($this_month);
    $data_prev = $this->get_dashboard_data($last_month);

    return response()->json([
      'total_sales' => $data['total_sales'],
      'total_sales_in_house'  => $data['total_sales_in_house'],
      'total_sales_part_time' => $data['total_sales_part_time'],
      'total_sales_dropship'  => $data['total_sales_dropship'],
      'total_sales_fighter' => $data['total_sales_fighter'],
      'total_gross_profit'  => $data['total_gross_profit'],
      'ad_cost' => $data['ad_cost'],
      'postage_cost'  => $data['postage_cost'],
      'nett_profit' => $data['nett_profit'],
      'sales_roi' => $data['sales_roi'],
      'profit_roi' => $data['profit_roi'],
      'total_sales_prev' => $data_prev['total_sales'],
      'total_sales_in_house_prev'  => $data_prev['total_sales_in_house'],
      'total_sales_part_time_prev' => $data_prev['total_sales_part_time'],
      'total_sales_dropship_prev'  => $data_prev['total_sales_dropship'],
      'total_sales_fighter_prev' => $data_prev['total_sales_fighter'],
      'total_gross_profit_prev'  => $data_prev['total_gross_profit'],
      'ad_cost_prev' => $data_prev['ad_cost'],
      'postage_cost_prev'  => $data_prev['postage_cost'],
      'nett_profit_prev' => $data_prev['nett_profit'],
      'sales_roi_prev' => $data_prev['sales_roi'],
      'profit_roi_prev' => $data_prev['profit_roi'],
    ]);
  }

  private function get_dashboard_data($date) {
    // $date = date('Y-m');

    $sales = DailySales::where('report_date', 'like', $date.'%')->get();
    $cost = DailyReportMarketing::where('report_date', 'like', $date.'%')->sum('daily_ads_spending');

    $total_sales = 0;
    $total_sales_in_house = 0;
    $total_sales_part_time = 0;
    $total_sales_dropship = 0;
    $total_sales_fighter = 0;
    $total_gross_profit = 0;

    foreach($sales as $sale) {
      $total_sales += $sale->sales_bizapp;
      $total_sales_in_house += $sale->sales_in_house;
      $total_sales_part_time += $sale->sales_part_time;
      $total_sales_dropship += $sale->sales_dropship;
      $total_sales_fighter += $sale->sales_fighter;
      $total_gross_profit += $sale->gross_profit;
    }

    $parcel_out = DailyReportParcel::where('report_date', 'like', $date.'%')->sum('qty');
    $parcel_repost = ParcelReturn::where('return_date', 'like', $date.'%')
      ->where('status', 'R')->count();

    $postage_cost = ($parcel_out + $parcel_repost) * 7;
    $nett_profit = $total_gross_profit - $postage_cost - $cost;

    return [
      'total_sales' => $total_sales,
      'total_sales_in_house'  => $total_sales_in_house,
      'total_sales_part_time' => $total_sales_part_time,
      'total_sales_dropship'  => $total_sales_dropship,
      'total_sales_fighter' => $total_sales_fighter,
      'total_gross_profit'  => $total_gross_profit,
      'ad_cost' => (float) $cost,
      'postage_cost'  => $postage_cost,
      'nett_profit' => $nett_profit,
      'sales_roi' => $cost > 0 ? ($total_sales_in_house + $total_sales_part_time) / $cost : ($total_sales_in_house + $total_sales_part_time) > 0 ? 100 : 0,
      'profit_roi' => $cost > 0 ? $nett_profit / $cost : $nett_profit > 0 ? 100 : 0
    ];
  }

  public function summary(Request $request) {
    $start = Carbon::now()->startOfMonth();
    $end = Carbon::now()->endOfMonth();

    if ($request->query('from_date') && $request->query('end_date')) {
      $start = Carbon::createFromFormat('Y-m-d', $request->query('from_date'));
      $end = Carbon::createFromFormat('Y-m-d', $request->query('end_date'));
    }

    $data = [];
    while($start->lte($end)) {
      $day = $start->format('Y-m-d');

      $sales = DailySales::where('report_date', $day)->get();
      $cost = DailyReportMarketing::where('report_date', $day)->sum('daily_ads_spending');

      $total_sales = 0;
      $total_sales_in_house = 0;
      $total_sales_part_time = 0;
      $total_sales_dropship = 0;
      $total_sales_fighter = 0;
      $total_gross_profit = 0;

      foreach($sales as $sale) {
        $total_sales += $sale->sales_bizapp;
        $total_sales_in_house += $sale->sales_in_house;
        $total_sales_part_time += $sale->sales_part_time;
        $total_sales_dropship += $sale->sales_dropship;
        $total_sales_fighter += $sale->sales_fighter;
        $total_gross_profit += $sale->gross_profit;
      }

      $parcel_out = DailyReportParcel::where('report_date', $day)->sum('qty');
      $parcel_repost = ParcelReturn::where('return_date', $day)
        ->where('status', 'R')->count();

      $postage_cost = ($parcel_out + $parcel_repost) * 7;
      $nett_profit = $total_gross_profit - $postage_cost - $cost;

      if ($cost > 0) {
        $sales_roi = $total_sales_in_house / $cost;
        $profit_roi = $nett_profit / $cost;
      } else {
        $sales_roi = $total_sales_in_house > 0 ? 100 : 0;
        $profit_roi = $nett_profit > 0 ? 100 : 0;
      }

      $data[] = [
        'date'  => $day,
        'total_sales' => $total_sales,
        'total_sales_in_house'  => $total_sales_in_house,
        'total_sales_part_time' => $total_sales_part_time,
        'total_sales_dropship'  => $total_sales_dropship,
        'total_sales_fighter' => $total_sales_fighter,
        'total_gross_profit'  => $total_gross_profit,
        'ad_cost' => (float) $cost,
        'postage_cost'  => $postage_cost,
        'nett_profit' => $nett_profit,
        'sales_roi' => $sales_roi,
        'profit_roi' => $profit_roi
      ];

      $start->addDay();
    }

    return response()->json($data);
  }

  public function weekly(Request $request) {
    $start = Carbon::now()->startOfWeek(Carbon::MONDAY);
    $end = Carbon::now()->endOfWeek(Carbon::SUNDAY);    

    $arr = [];

    while($start->lte($end)) {
      $day = $start->format('Y-m-d');

      $sales = DailySales::where('report_date', $day)
        ->get();

      $total_sales = 0;
      $total_sales_in_house = 0;
      $total_sales_part_time = 0;
      $total_sales_dropship = 0;
      $total_sales_fighter = 0;
      $total_gross_profit = 0;

      foreach($sales as $sale) {
        $total_sales += $sale->sales_bizapp;
        $total_sales_in_house += $sale->sales_in_house;
        $total_sales_part_time += $sale->sales_part_time;
        $total_sales_dropship += $sale->sales_dropship;
        $total_sales_fighter += $sale->sales_fighter;
        $total_gross_profit += $sale->gross_profit;
      }

      $arr[$day] = [
        'sales_bizapp'  => $total_sales,
        'sales_in_house'  => $total_sales_in_house,
        'sales_part_time' => $total_sales_part_time,
        'sales_fighter' => $total_sales_fighter,
        'sales_dropship'  => $total_sales_dropship,
        'gross_profit'  => $total_gross_profit
      ];
      $start->addDay();
    }

    return response()->json($arr);
  }
  public function monthly(Request $request) {
    $start = Carbon::now()->startOfMonth();
    $end = Carbon::now()->endOfMonth();    

    $arr = [];

    while($start->lte($end)) {
      $day = $start->format('Y-m-d');

      $sales = DailySales::where('report_date', $day)
        ->get();

      $total_sales = 0;
      $total_sales_in_house = 0;
      $total_sales_part_time = 0;
      $total_sales_dropship = 0;
      $total_sales_fighter = 0;
      $total_gross_profit = 0;

      foreach($sales as $sale) {
        $total_sales += $sale->sales_bizapp;
        $total_sales_in_house += $sale->sales_in_house;
        $total_sales_part_time += $sale->sales_part_time;
        $total_sales_dropship += $sale->sales_dropship;
        $total_sales_fighter += $sale->sales_fighter;
        $total_gross_profit += $sale->gross_profit;
      }

      $arr[$day] = [
        'sales_bizapp'  => $total_sales,
        'sales_in_house'  => $total_sales_in_house,
        'sales_part_time' => $total_sales_part_time,
        'sales_fighter' => $total_sales_fighter,
        'sales_dropship'  => $total_sales_dropship,
        'gross_profit'  => $total_gross_profit
      ];
      $start->addDay();
    }

    return response()->json($arr);
  }

  public function yearly(Request $request) {
    $start = Carbon::now()->startOfYear();
    $end = Carbon::now()->endOfYear();    

    $arr = [];

    while($start->lte($end)) {
      $day = $start->format('Y-m');

      $sales = DailySales::where('report_date', 'like', $day.'%')
        ->get();

      $total_sales = 0;
      $total_sales_in_house = 0;
      $total_sales_part_time = 0;
      $total_sales_dropship = 0;
      $total_sales_fighter = 0;
      $total_gross_profit = 0;

      foreach($sales as $sale) {
        $total_sales += $sale->sales_bizapp;
        $total_sales_in_house += $sale->sales_in_house;
        $total_sales_part_time += $sale->sales_part_time;
        $total_sales_dropship += $sale->sales_dropship;
        $total_sales_fighter += $sale->sales_fighter;
        $total_gross_profit += $sale->gross_profit;
      }

      $arr[$day] = [
        'sales_bizapp'  => $total_sales,
        'sales_in_house'  => $total_sales_in_house,
        'sales_part_time' => $total_sales_part_time,
        'sales_fighter' => $total_sales_fighter,
        'sales_dropship'  => $total_sales_dropship,
        'gross_profit'  => $total_gross_profit
      ];
      $start->addMonth();
    }

    return response()->json($arr);
  }

  public function analysis(Request $request) {
    $start = Carbon::now()->subDay(14);
    $end = Carbon::now();

    if ($request->query('from_date') && $request->query('end_date')) {
      $start = Carbon::createFromFormat('Y-m-d', $request->query('from_date'));
      $end = Carbon::createFromFormat('Y-m-d', $request->query('end_date'));
    }

    $data = [];
    while($start->lte($end)) {
      $day = $start->format('Y-m-d');
      $total_sales = 0;
      $total_ads = 0;
      $sales = DailySales::where('report_date', $day)
        ->get();
      foreach($sales as $sale) {
        $total_sales += $sale->sales_bizapp;

      }
      $ads = DailyReportMarketing::where('report_date', $day)
        ->get();
      foreach($ads as $ad) {
        $total_ads += $ad->daily_ads_spending;
      }
      $data[$day] = [
        'total_sales' => $total_sales,
        'total_ads' => $total_ads
      ];
      $start->addDay();
    }

    return response()->json($data);
  }

  public function top_seller(Request $request) {
    $date = Carbon::now();
    $limit = $request->query('limit') ? (int) $request->query('limit') : 3;

    $sales = DailyReportSales::selectRaw('*, sum(lead_bizapp_paid_amount + lead_rebut_paid_amount + lead_direct_paid_amount) as amount')
      ->where('report_date', 'like', $date->format('Y-m').'%')
      ->with(['user'])->groupBy('user_id')
      ->orderBy('amount','desc')->limit($limit)->get();

    $arr = [];

    foreach($sales as $sale) {
      $arr[] = [
        'name'  => $sale->user->name,
        'amount' => (float) $sale->amount 
      ];
    }

    return response()->json($arr);
  }

  public function top_fighters(Request $request) {
    $date = Carbon::now();
    $limit = $request->query('limit') ? (int) $request->query('limit') : 3;

    $sales = SalesFighter::selectRaw('*, sum(sales) as amount')
      ->where('report_date', 'like', $date->format('Y-m').'%')
      ->with(['user'])->groupBy('user_id')
      ->orderBy('amount','desc')->limit($limit)->get();

    $arr = [];

    foreach($sales as $sale) {
      $arr[] = [
        'name'  => $sale->user->name,
        'amount' => (float) $sale->amount 
      ];
    }

    return response()->json($arr);
  }
}