<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\DailyReportSales;
use App\DailySummarySales;
use App\User;
use Carbon\Carbon;

class DailyReportSalesController extends Controller
{

  public function index(Request $request) {
    if(app('auth')->user()->hasRole('Superadmin')) {
      $data = DailyReportSales::with(['product', 'user']);
    } else {
      $data = DailyReportSales::where('user_id', app('auth')->user()->id)
        ->with(['product', 'user']);
    }

    if($request->query('from_date') && $request->query('end_date')) {
      $data = $data->whereBetween('report_date', [$request->query('from_date'), $request->query('end_date')]);
    }
    $data = $data->orderBy('report_date','desc')->get();
    return response()->json($data);
  }

  public function store(Request $request) {
    $params = $request->all();
    $rules = DailyReportSales::$rules;

    $params['user_id'] = app('auth')->user()->hasRole('Superadmin') ? $params['user_id'] : app('auth')->user()->id;

    $rules['report_date'] = [
      'required', 
      Rule::unique('daily_report_sales')->where(function ($query) use ($params) {
        return $query->where('product_id', $params['product_id'] )
                ->where('user_id', $params['user_id'])
                ->where('report_date', $params['report_date']);
      })
    ];
    $this->validate($request, $rules);

    $data = DailyReportSales::create($params);
    return response()->json($data, 201);
  }

  public function show($id) {
    return response()->json(DailyReportSales::with(['product', 'user'])->findOrFail($id));
  }

  public function update($id, Request $request) {
    $data = DailyReportSales::findOrFail($id);

    if(!app('auth')->user()->hasRole('Superadmin')) {
      if ($data->user_id != app('auth')->user()->id) {
        return response()->json(['error' => 'Unauthorized - Cannot change other\'s data.'], 401);
      }
    } 
    $user_id = app('auth')->user()->hasRole('Superadmin') ? $data->user_id : app('auth')->user()->id;

    $rules = DailyReportSales::$rules;
    $rules['report_date'] = [
      'required', 
      Rule::unique('daily_report_sales')->where(function ($query) use ($id, $data, $user_id) {
        return $query->where('id', '<>', $id)
                    ->where('user_id', $user_id)
                    ->where('product_id', $data->product_id)
                    ->where('report_date', $data->report_date);
      })
    ];
    $this->validate($request, $rules);
    $params = $request->all();
    $data->update($params);
    return response()->json($data, 200);
  }

  public function destroy($id) {
    $data = DailyReportSales::with(['product', 'user'])->findOrFail($id);

    if(!app('auth')->user()->hasRole('Superadmin')) {
      if($data->user_id != app('auth')->user()->id) {
        return response()->json(['message' => 'Unauthorized - Cannot delete other\'s data.'], 401);
      }
    }
    
    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }

  public function weekly(Request $request) {
    $start = Carbon::now()->startOfWeek(Carbon::MONDAY);
    $end = Carbon::now()->endOfWeek(Carbon::SUNDAY);

    $arr = [];

    while($start->lte($end)) {
      $day = $start->format('Y-m-d');

      $sales = DailyReportSales::where('report_date', $day)->with(['product']);

      if(app('auth')->user()->hasRole('Sales')) {
        $sales = $sales->where('user_id', app('auth')->user()->id);
      }

      $sales = $sales->get();

      if (count($sales) > 0) {
        foreach($sales as $sale) {
          $total_sales = $sale->lead_bizapp_paid_amount + $sale->lead_rebut_paid_amount + $sale->lead_direct_paid_amount;
          
          if (array_key_exists($sale->product->code, $arr)) {
            $arr[$sale->product->code][$day] = $total_sales;
          } else {
            $arr[$sale->product->code] = [
              $day => $total_sales
            ];
          }
        }
      }
      $start->addDay();
    }

    $data_arr = [];
    foreach($arr as $k => $v) {
      $start = Carbon::now()->startOfWeek(Carbon::MONDAY);
      $end = Carbon::now()->endOfWeek(Carbon::SUNDAY);

      //$day_arr = [];
      while($start->lte($end)) {
        $day = $start->format('Y-m-d');
        
        if (!array_key_exists($day, $data_arr)) {
          $data_arr[$day] = [
            $k => isset($v[$day]) ? $v[$day] : 0
          ];
        } else {
          $data_arr[$day][$k] = isset($v[$day]) ? $v[$day] : 0;
        }
        // if (array_key_exists($day, $v)) {
        //   $day_arr[$day] = $v[$day];
        // } else {
        //   $day_arr[$day] = 0;
        // }

        $start->addDay();
      }
      //$data_arr[$k] = $day_arr;
    }

    return response()->json($data_arr);
  }

  public function dashboard(Request $request) {
    $end = Carbon::now()->endOfMonth();
    $start = Carbon::now()->startOfMonth();
    $end_prev = Carbon::now()->subMonth()->endOfMonth();
    $start_prev = Carbon::now()->subMonth()->startOfMonth();

    $arr = $this->get_dashboard_data($start, $end);
    $arr_prev = $this->get_dashboard_data($start_prev, $end_prev);

    return response()->json([
      // 'total_sales'   => $arr['total_sales'],
      // 'commission'    => $arr['commission'],
      // 'total_leads_follow_up' => $arr['total_leads_follow_up'],
      'total_leads'   => $arr['total_leads'],
      'total_close_leads' => $arr['total_close_leads'],
      'avg_closing'  => $arr['avg_closing'],
      'avg_closing_bizapp'  => $arr['avg_closing_bizapp'],
      'avg_closing_rebut'  => $arr['avg_closing_rebut'],
      'avg_closing_direct'  => $arr['avg_closing_direct'],
      'avg_closing_sales'  => $arr['avg_closing_sales'],
      'total_sales_repeat'  => $arr['total_sales_repeat'],
      'total_sales_unpaid'  => $arr['total_sales_unpaid'],
      'total_follow_up' => $arr['total_follow_up'],
      'total_payment' => $arr['total_payment'],
      'total_customers' => $arr['total_customers'],
      // 'total_sales_prev'   => $arr_prev['total_sales'],
      // 'commission_prev'    => $arr_prev['commission'],
      // 'total_leads_follow_up_prev' => $arr_prev['total_leads_follow_up'],
      'total_leads_prev'   => $arr_prev['total_leads'],
      'total_close_leads_prev' => $arr_prev['total_close_leads'],
      'avg_closing_prev'  => $arr_prev['avg_closing'],
      'avg_closing_bizapp_prev'  => $arr_prev['avg_closing_bizapp'],
      'avg_closing_rebut_prev'  => $arr_prev['avg_closing_rebut'],
      'avg_closing_direct_prev'  => $arr_prev['avg_closing_direct'],
      'avg_closing_sales_prev'  => $arr_prev['avg_closing_sales'],
      'total_sales_repeat_prev'  => $arr_prev['total_sales_repeat'],
      'total_sales_unpaid_prev'  => $arr_prev['total_sales_unpaid'],
      'total_follow_up_prev' => $arr_prev['total_follow_up'],
      'total_payment_prev' => $arr_prev['total_payment'],
      'total_customers_prev' => $arr_prev['total_customers']
      // 'total_sales_repeat_order_prev'  => $arr_prev['total_sales_repeat_order'],
      // 'total_sales_unpaid_prev'  => $arr_prev['total_sales_unpaid']
    ]);
  }

  public function summary(Request $request) {
    $date = date('Y-m-d');
    if ($request->query('date')) {
      $date = $request->query('date');
    }
    $users = User::role('Sales')->orderBy('name', 'asc')->get();

    $data = [];
    foreach($users as $user) {
      $sales = DailyReportSales::where('report_date', $date)
        ->where('user_id', $user->id)
        ->with(['product'])->get();
      
      $arr = [];
      $total_bizapp = 0;
      $total_bizapp_close = 0;
      $total_bizapp_close_amt = 0;
      $total_bizapp_payment = 0;
      $total_rebut = 0;
      $total_rebut_close = 0;
      $total_rebut_close_amt = 0;
      $total_rebut_payment = 0;
      $total_direct = 0;
      $total_direct_close = 0;
      $total_direct_close_amt = 0;
      $total_direct_payment = 0;
      $total_payment = 0;
      $total_sales_repeat = 0;
      $total_follow_up = 0;

      foreach($sales as $sale) {
        if (array_key_exists($sale->product->code, $arr)) {
          $arr[$sale->product->code] += $sale->lead_bizapp_paid_qty + 
            $sale->lead_rebut_paid_qty + $sale->lead_direct_paid_qty;
        } else {
          $arr[$sale->product->code] = $sale->lead_bizapp_paid_qty + 
            $sale->lead_rebut_paid_qty + $sale->lead_direct_paid_qty;
        }
        $total_bizapp += $sale->lead_bizapp_total;
        $total_bizapp_close += $sale->lead_bizapp_close;
        $total_bizapp_close_amt += $sale->lead_bizapp_close_amount;
        $total_bizapp_payment += $sale->lead_bizapp_paid_amount;
        $total_rebut += $sale->lead_rebut_total;
        $total_rebut_close += $sale->lead_rebut_close;
        $total_rebut_close_amt += $sale->lead_rebut_close_amount;
        $total_rebut_payment += $sale->lead_rebut_paid_amount;
        $total_direct += $sale->lead_direct_total;
        $total_direct_close += $sale->lead_direct_close;
        $total_direct_close_amt += $sale->lead_direct_close_amount;
        $total_direct_payment += $sale->lead_direct_paid_amount;
      }

      $summary = DailySummarySales::where('report_date', $date)
        ->where('user_id', $user->id)->first();
      if ($summary) {
        $total_payment = $summary->total_payment_collected;
        $total_sales_repeat = $summary->sales_repeat_order;
        $total_follow_up = $summary->total_follow_up;
      }
      $prod = [];
      foreach($arr as $k => $v) {
        $prod[]= $v.' '.$k;
      }
      $data[] = [
        'date'  => $date,
        'seller'  => $user->name,
        'total_bizapp'  => $total_bizapp,
        'total_bizapp_close'  => $total_bizapp_close,
        'total_bizapp_close_amt'  => $total_bizapp_close_amt,
        'total_bizapp_payment'  => $total_bizapp_payment,
        'total_bizapp_cr' => $total_bizapp > 0 ? ($total_bizapp_close/$total_bizapp) * 100 : 0,
        'total_rebut'  => $total_rebut,
        'total_rebut_close'  => $total_rebut_close,
        'total_rebut_close_amt'  => $total_rebut_close_amt,
        'total_rebut_payment'  => $total_rebut_payment,
        'total_rebut_cr' => $total_rebut > 0 ? ($total_rebut_close/$total_rebut) * 100 : 0,
        'total_direct'  => $total_direct,
        'total_direct_close'  => $total_direct_close,
        'total_direct_close_amt'  => $total_direct_close_amt,
        'total_direct_payment'  => $total_direct_payment,
        'total_direct_cr' => $total_direct > 0 ? ($total_direct_close/$total_direct) * 100 : 0,
        'total_payment' => $total_payment,
        'total_sales_repeat'  => $total_sales_repeat,
        'total_follow_up' => $total_follow_up,
        'products'  => implode(', ', $prod)
      ];
    }

    return response()->json($data);
  }

  private function get_dashboard_data($start, $end) {
    //$end = Carbon::now()->endOfMonth();
    //$start = Carbon::now()->startOfMonth();
    $arr = [];
    $total_sales = 0;
    $total_bizapp_sales = 0;
    $total_rebut_sales = 0;
    $total_direct_sales = 0;
    $total_leads = 0;
    $total_close_leads = 0;
    $total_bizapp_leads = 0;
    $total_rebut_leads = 0;
    $total_direct_leads = 0;
    $total_bizapp_close_leads = 0;
    $total_rebut_close_leads = 0;
    $total_direct_close_leads = 0;
    $total_bizapp_paid_qty = 0;
    $total_rebut_paid_qty = 0;
    $total_direct_paid_qty = 0;
    $total_follow_up = 0;
    $total_sales_repeat = 0;
    $total_sales_unpaid = 0;
    $total_payment = 0;
    $total_customers = 0;

    while($start->lte($end)) {
      // info($last_week->format('D'));
      $sum_avg = 0;
      $day = $start->format('Y-m-d');

      $sales = DailyReportSales::where('report_date', $day)->with(['product']);
      $summary = DailySummarySales::where('report_date', $day);
      if(app('auth')->user()->hasRole('Sales')) {
        $sales = $sales->where('user_id', app('auth')->user()->id);
        $summary = $summary->where('user_id', app('auth')->user()->id);
      }

      $sales = $sales->get();
      $summary = $summary->get();

      if (count($sales) > 0) {
        foreach($sales as $sale) {
          $total_bizapp_sales += $sale->lead_bizapp_paid_amount;
          $total_rebut_sales += $sale->lead_rebut_paid_amount;
          $total_direct_sales += $sale->lead_direct_paid_amount;
          $total_bizapp_leads += $sale->lead_bizapp_total;
          $total_rebut_leads += $sale->lead_rebut_total;
          $total_direct_leads += $sale->lead_direct_total;
          $total_bizapp_close_leads += $sale->lead_bizapp_close;
          $total_rebut_close_leads += $sale->lead_rebut_close;
          $total_direct_close_leads += $sale->lead_direct_close;
          $total_customers += $sale->lead_bizapp_paid + $sale->lead_rebut_paid + $sale->lead_direct_paid;
        }
      }

      foreach($summary as $s) {
        $total_follow_up += $s->total_follow_up;
        $total_sales_repeat += $s->sales_repeat_order;
        $total_sales_unpaid += $s->sales_unpaid;
        $total_payment += $s->total_payment_collected;
      }
      
      $start->addDay();
    }

    //$total_sales = $total_bizapp_sales + $total_rebut_sales + $total_direct_sales;
    $total_leads = $total_bizapp_leads + $total_rebut_leads + $total_direct_leads;
    $total_close_leads = $total_bizapp_close_leads + $total_rebut_close_leads + $total_direct_close_leads;
    //$commission = $total_sales * 0.02;
    $avg_closing_rate = $total_leads > 0 ? ($total_close_leads/$total_leads) * 100 : 0;
    $avg_closing_bizapp = $total_bizapp_leads > 0 ? ($total_bizapp_close_leads/$total_bizapp_leads) * 100 : 0;
    $avg_closing_rebut = $total_rebut_leads > 0 ? ($total_rebut_close_leads/$total_rebut_leads) * 100 : 0;
    $avg_closing_direct = $total_direct_leads > 0 ? ($total_direct_close_leads/$total_direct_leads) * 100 : 0;
    $avg_closing_sales = $total_close_leads > 0 ? ($total_customers / $total_close_leads) * 100 : 0;

    return [
      //'commission'    => $commission,
      'total_leads'   => $total_leads,
      'total_close_leads' => $total_close_leads,
      'avg_closing'  => $avg_closing_rate,
      'avg_closing_bizapp' => $avg_closing_bizapp,
      'avg_closing_rebut' => $avg_closing_rebut,
      'avg_closing_direct'  => $avg_closing_direct,
      'avg_closing_sales' => $avg_closing_sales,
      'total_follow_up' => $total_follow_up,
      'total_sales_repeat'  => $total_sales_repeat,
      'total_sales_unpaid'  => $total_sales_unpaid,
      'total_payment' => $total_payment,
      'total_customers' => $total_customers
    ];
  }
}