<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\DailyReportParcel;
use App\ParcelReturn;
use Carbon\Carbon;

class DailyReportParcelController extends Controller
{

  public function index(Request $request) {
    $data = DailyReportParcel::orderBy('report_date', 'desc');

    if($request->query('from_date') && $request->query('end_date')) {
      $data = $data->whereBetween('report_date', [$request->query('from_date'), $request->query('end_date')]);
    }
    // if($request->query('from_date_return') && $request->query('end_date_return')) {
    //   $data = $data->whereBetween('return_date', [$request->query('from_date_return'), $request->query('end_date_return')]);
    // }
    $data = $data->get();
    return response()->json($data);
  }

  public function store(Request $request) {
    $params = $request->all();
    $rules = DailyReportParcel::$rules;

    $this->validate($request, $rules);
    // if (app('auth')->user()->hasRole('Marketing')) {
    //   $params['user_id'] = app('auth')->user()->id;
    // }

    $data = DailyReportParcel::create($params);
    return response()->json($data, 201);
  }

  public function show($id) {
    return response()->json(DailyReportParcel::findOrFail($id));
  }

  public function update($id, Request $request) {
    $data = DailyReportParcel::findOrFail($id);
    $params = $request->all();

    $rules = DailyReportParcel::$rules;

    $rules['report_date'] = [
      'required', 
      Rule::unique('daily_report_parcel')->where(function ($query) use ($id, $params) {
        return $query->where('id', '<>', $id)
                    ->where('report_date', $params['report_date']);
      })
    ];
    
    $this->validate($request, $rules);
    $params = $request->all();
    $data->update($params);
    return response()->json($data, 200);
  }

  public function destroy($id) {
    $data = DailyReportParcel::findOrFail($id);
    
    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }

  public function dashboard(Request $request) {
    $end = Carbon::now()->endOfMonth();
    $start = Carbon::now()->startOfMonth();
    $end_prev = Carbon::now()->subMonth()->endOfMonth();
    $start_prev = Carbon::now()->subMonth()->startOfMonth();

    $arr = $this->get_dashboard_data($start, $end);
    $arr_prev = $this->get_dashboard_data($start_prev, $end_prev);

    return response()->json([
      'total_parcel_out'    => $arr['total_parcel_out'],
      'total_parcel_return' => $arr['total_parcel_return'],
      'total_parcel_repost' => $arr['total_parcel_repost'],
      'total_est_postage' => $arr['total_est_postage'],
      'total_parcel_out_prev' => $arr_prev['total_parcel_out'],
      'total_parcel_return_prev'  => $arr_prev['total_parcel_return'],
      'total_parcel_repost_prev' => $arr_prev['total_parcel_repost'],
      'total_est_postage_prev' => $arr_prev['total_est_postage'],
    ]);
  }

  private function get_dashboard_data($start, $end) {
    //$end = Carbon::now()->endOfMonth();
    //$start = Carbon::now()->startOfMonth();
    $total_parcel_out = 0;
    $total_parcel_return = 0;
    $total_parcel_repost = 0;

    while($start->lte($end)) {
      $day = $start->format('Y-m-d');

      $parcels = DailyReportParcel::where('report_date', $day)->get();

      if (count($parcels) > 0) {
        foreach($parcels as $parcel) {
          $total_parcel_out += $parcel->qty;
        }
      }

      $returns = ParcelReturn::where('return_date', $day)
          ->where('status', '<>', 'R')->get();
      $total_parcel_return += count($returns);
      
      $repost = ParcelReturn::where('return_date', $day)
          ->where('status', 'R')->get();
      $total_parcel_repost += count($repost);

      $start->addDay();
    }

    return [
      //'commission'    => $commission,
      'total_parcel_out'    => $total_parcel_out,
      'total_parcel_return' => $total_parcel_return,
      'total_parcel_repost' => $total_parcel_repost,
      'total_est_postage' => ($total_parcel_out + $total_parcel_repost) * 7
    ];
  }

  public function weekly(Request $request) {
    $start = Carbon::now()->startOfWeek(Carbon::MONDAY);
    $end = Carbon::now()->endOfWeek(Carbon::SUNDAY);

    $arr = [];

    while($start->lte($end)) {
      $day = $start->format('Y-m-d');
      $parcel_out = 0;
      $parcel_in = 0;

      $parcels = DailyReportParcel::where('report_date', $day)->get();

      if (count($parcels) > 0) {
        foreach($parcels as $parcel) {
          $parcel_out += $parcel->qty;
        }
      }
      $returns = ParcelReturn::where('return_date', $day)
        ->where('status', '<>', 'R')->get();
      $arr[$day] = [
        'parcel_in' => count($returns),
        'parcel_out'=> $parcel_out
      ];
      $start->addDay();
    }

    return response()->json($arr);
  }

  public function monthly(Request $request) {
    $start = Carbon::now()->startOfMonth();
    $end = Carbon::now()->endOfMonth();

    $arr = [];

    while($start->lte($end)) {
      $day = $start->format('Y-m-d');
      $parcel_out = 0;
      $parcel_in = 0;

      $parcels = DailyReportParcel::where('report_date', $day)->get();

      if (count($parcels) > 0) {
        foreach($parcels as $parcel) {
          $parcel_out += $parcel->qty;
        }
      }
      $returns = ParcelReturn::where('return_date', $day)
        ->where('status', '<>', 'R')->get();
      $arr[$day] = [
        'parcel_in' => count($returns),
        'parcel_out'=> $parcel_out
      ];
      $start->addDay();
    }

    return response()->json($arr);
  }

  public function yearly(Request $request) {
    $start = Carbon::now()->startOfYear();
    $end = Carbon::now()->endOfYear();

    $arr = [];

    while($start->lte($end)) {
      $day = $start->format('Y-m');
      $parcel_out = 0;
      $parcel_in = 0;

      $parcels = DailyReportParcel::where('report_date', 'like', $day.'%')->get();

      if (count($parcels) > 0) {
        foreach($parcels as $parcel) {
          $parcel_out += $parcel->qty;
        }
      }
      $returns = ParcelReturn::where('return_date', 'like', $day.'%')
        ->where('status', '<>', 'R')->get();
      $arr[$day] = [
        'parcel_in' => count($returns),
        'parcel_out'=> $parcel_out
      ];
      $start->addMonth();
    }

    return response()->json($arr);
  }

  public function summary(Request $request) {
    $start = Carbon::now()->startOfMonth();
    $end = Carbon::now()->endOfMonth();

    if ($request->query('from_date') && $request->query('end_date')) {
      $start = Carbon::createFromFormat('Y-m-d', $request->query('from_date'));
      $end = Carbon::createFromFormat('Y-m-d', $request->query('end_date'));
    }

    $data = [];
    while($start->lte($end)) {
      $day = $start->format('Y-m-d');
      $parcel_out = 0;
      $parcel_in = 0;
      $parcel_repost = 0;

      $parcels = DailyReportParcel::where('report_date', $day)->get();

      if (count($parcels) > 0) {
        foreach($parcels as $parcel) {
          $parcel_out += $parcel->qty;
        }
      }
      $returns = ParcelReturn::where('return_date', $day)
        ->where('status', '<>', 'R')->get();
      $repost = ParcelReturn::where('return_date', $day)
        ->where('status', 'R')->get();
      $parcel_repost = count($repost);

      $data[] = [
        'date'  => $day,
        'parcel_return' => count($returns),
        'parcel_out'=> $parcel_out,
        'parcel_repost' => $parcel_repost,
        'est_postage' => ($parcel_out + $parcel_repost) * 7
      ];
      $start->addDay();
    }

    return response()->json($data);
  }
}