<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\CampaignLead;
use App\Campaign;
use App\CampaignCost;
use App\WorkingDay;
use App\Attendance;
use App\Setting;
use Carbon\Carbon;

class CampaignLeadController extends Controller
{

  public function index(Request $request) {
    if(app('auth')->user()->hasRole('Admin')) {
      $data = CampaignLead::with(['campaign', 'product', 'sales', 'assignee', 'creator']);
    } else if(app('auth')->user()->hasRole('Sales')) {
      $data = CampaignLead::where('assigned_to', app('auth')->user()->id)
        ->with(['campaign', 'product', 'sales', 'assignee', 'creator']);
    } else {
      $data = CampaignLead::where('created_by', app('auth')->user()->id)
        ->with(['campaign', 'product', 'sales', 'assignee', 'creator']);
    }
    if($request->query('from_date') && $request->query('end_date')) {
      $data = $data->whereBetween('assigned_date', [$request->query('from_date'), $request->query('end_date')]);
    }
    if($request->query('date')) {
      $data = $data->where('assigned_date', 'like', $request->query('date').'%');
    }
    $data = $data->orderBy('assigned_date','desc')->get();
    return response()->json($data);
  }

  public function store(Request $request) {
    $params = $request->all();
    $campaign = Campaign::findOrFail($params['campaign_id']);
    $rules = CampaignLead::$rules;
    $rules['phone'] = [
      'required',
      'numeric',
      function ($attribute, $value, $fail) use ($campaign) {
        $yesterday = Carbon::now()->subDay()->format("Y-m-d");
        $start_date = Carbon::createFromFormat('Y-m-d H:i:s', $campaign->created_at)->format("Y-m-d");
        
        if(Carbon::now()->format('Y-m-d') > $start_date){
          $cost = CampaignCost::where('campaign_id', $campaign->id)
                ->where('is_confirmed', true)
                ->where('cost_date', $yesterday)->get();
          if(count($cost) < 1) {
            return $fail('Please insert ad cost for '.$yesterday.' before add leads');
          }
        }

        $lead = CampaignLead::where('phone', $value)
                ->with(['campaign', 'campaign.creator'])
                ->orderBy('created_at', 'desc')->first();
        if($lead) {
          $settings = Setting::where('key', 'phone_validity_period')->first();
          if($lead->created_at > Carbon::now()->subDays((int) $settings->value)->format("Y-m-d H:i:s")) {
            return $fail('Duplicate phone no. used by other campaign: '.$lead->campaign->name.' ('.$lead->campaign->creator->name.')');
          }
        }
      }
    ];
    $this->validate($request, $rules);
    $params['created_by'] = app('auth')->user()->id;
    $params['product_id'] = $campaign->product_id;

    $data = CampaignLead::create($params);
    return response()->json($data, 201);
  }

  public function show($id) {
    return response()->json(CampaignLead::with(['campaign', 'assignee', 'creator'])->findOrFail($id));
  }

  public function update($id, Request $request) {
    $data = CampaignLead::with(['campaign'])->findOrFail($id);

    if(!app('auth')->user()->hasRole('Admin')) {
      if ($data->created_by != app('auth')->user()->id) {
        return response()->json(['error' => 'Unauthorized - Cannot change other\'s data.'], 401);
      }
    } 

    if ($data->is_assigned) {
      return response()->json(['error' => 'Unauthorized - Cannot change data if lead already assigned.'], 401);
    }
    $user_id = app('auth')->user()->hasRole('Admin') ? $data->id : app('auth')->user()->id;

    $rules = CampaignLead::$rules;
    $rules['phone'] = [
      'required', 
      'numeric',
      function ($attribute, $value, $fail) use ($id, $data) {
        $yesterday = Carbon::now()->subDay()->format("Y-m-d");
        $start_date = Carbon::createFromFormat('Y-m-d H:i:s', $data->campaign->created_at)->format("Y-m-d");

        if (Carbon::now()->format('Y-m-d') > $start_date) {
          $cost = CampaignCost::where('campaign_id', $data->campaign_id)
                ->where('is_confirmed', true)
                ->where('cost_date', $yesterday)->get();
          if(count($cost) < 1) {
            return $fail('Please insert ad cost for '.$yesterday.' before add leads');
          }
        }

        $lead = CampaignLead::where('phone', $value)
                ->where('id', '<>', $id)
                ->with(['campaign', 'campaign.creator'])
                ->orderBy('created_at', 'desc')->first();
        if($lead) {
          $settings = Setting::where('key', 'phone_validity_period')->first();
          if($lead->created_at > Carbon::now()->subDays((int) $settings->value)->format("Y-m-d H:i:s")) {
            return $fail('Duplicate phone no. used by other campaign: '.$lead->campaign->name.' ('.$lead->campaign->creator->name.')');
          }
        }
      }
    ];
    $this->validate($request, $rules);
    $params = $request->all();
    $data->update($params);
    return response()->json($data, 200);
  }

  public function destroy($id) {
    $data = CampaignLead::findOrFail($id);

    if(!app('auth')->user()->hasRole('Admin')) {
      if($data->created_by != app('auth')->user()->id) {
        return response()->json(['message' => 'Unauthorized - Cannot delete other\'s data.'], 401);
      }
    }
    if($data->is_assigned) {
      return response()->json(['message' => 'Unable to delete already assigned data'], 403);
    }
    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }

  public function generate() {
    $today = Carbon::now()->format('Y-m-d');
    $working_day = Attendance::where('user_id', app('auth')->user()->id)
      ->where('date', $today)->where('is_absent', false)->first();

    $total_leads = CampaignLead::where('is_assigned', false)->get();
    $total_attend = Attendance::where('date', $today)->where('is_absent', false)->get();

    $daily_lead = Setting::where('key', 'sales_lead_daily')->first();
    $daily_limit = (int) $daily_lead->value + (int) $working_day->total_boost;
    $limit =  ceil(count($total_leads)/count($total_attend));
    $limit = $limit > $daily_limit ? $daily_limit : $limit;
    
    if (!$working_day) {
      return response()->json(['message' => 'You are unable to generate leads because you flagged as absence.'], 403);
    }

    $exists = CampaignLead::where('assigned_date', 'like', $today.'%')->where('assigned_to', app('auth')->user()->id)->get();

    if (count($exists) >= $daily_limit) {
      return response()->json(['message' => 'You have reached maximum leads for today'], 403);
    }
    $final_limit = 0;
    if ($limit < $daily_limit) {
      $balance = abs($daily_limit - count($exists));
      $final_limit = $limit >= $balance ? $balance : $limit;
    } else {
      $final_limit = abs($limit - count($exists));
    }

    $leads = CampaignLead::where('is_assigned', false)->limit($final_limit)->update([
      'assigned_date' => Carbon::now(),
      'is_assigned' => true,
      'assigned_to' => app('auth')->user()->id
    ]);

    if ($leads == 0) {
      return response()->json(['message' => 'Insufficient leads. Please try later.'], 403);
    }
    $working_day->total_leads = $working_day->total_leads + $leads;
    $working_day->save();

    return response()->json(['message' => 'Leads generated successfully'], 200);
  }

  public function boost(Request $request) {
    $params = $request->all();
    if(!isset($params['user_id'])) {
      return response()->json(['message' => 'Missing user id'], 403);
    }
    $today = Carbon::now()->format('Y-m-d');
    $working_day = Attendance::where('user_id', $params['user_id'])
      ->where('date', $today)->where('is_absent', false)->first();
    info($working_day);
    if(!$working_day){
      return response()->json(['message' => 'You are unable to boost leads because you flagged user as absence.'], 403);
    }
    
    $total_leads = CampaignLead::where('is_assigned', false)->get();
    $total_attend = Attendance::where('date', $today)->where('is_absent', false)->get();
    $limit =  ceil(count($total_leads)/count($total_attend));
    $setting = Setting::where('key', 'sales_lead_boost')->first();

    $limit = $limit > (int) $setting->value ? (int) $setting->value : $limit;

    if(count($total_leads) > 0){
      if($working_day->total_leads <= (int) $working_day->default_leads_limit + (int) $working_day->total_boost) {
        return response()->json(['message' => 'You cannot boost unless max leads limit has been reached.'], 403);
      } else {
        $working_day->total_boost = $working_day->total_boost + $limit;
        $working_day->save();
      } 
    } else {
      return response()->json(['message' => 'Unable to boost lead. Please try later'], 403);
    }

    // $leads = CampaignLead::where('is_assigned', false)->limit($limit)->update([
    //   'assigned_date' => Carbon::now(),
    //   'is_assigned' => true,
    //   'assigned_to' => $params['user_id']
    // ]);

    // if ($leads == 0) {
    //   return response()->json(['message' => 'No lead boost generated for '.$today], 403);
    // }

    return response()->json(['message' => 'Leads boost successfully'], 200);
  }
}