<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\CampaignCost;
use App\Campaign;
use Carbon\Carbon;

class CampaignCostController extends Controller
{

  public function index(Request $request) {
    if(app('auth')->user()->hasRole('Admin')) {
      $data = CampaignCost::with(['campaign', 'creator']);
    } else {
      $data = CampaignCost::where('created_by', app('auth')->user()->id)
        ->with(['campaign', 'creator']);
    }

    if($request->query('from_date') && $request->query('end_date')) {
      $data = $data->whereBetween('cost_date', [$request->query('from_date'), $request->query('end_date')]);
    }
    $data = $data->get();
    return response()->json($data);
  }

  public function weekly(Request $request) {
    $now = Carbon::now();
    $last_week = Carbon::now()->subWeek();


    $arr = [];
    while($last_week->lte($now)) {
      // info($last_week->format('D'));
      $day = $last_week->format('D');

      $costs = CampaignCost::where('cost_date', $last_week->format('Y-m-d'))
                ->where('is_confirmed', true);

      if(app('auth')->user()->hasRole('Marketing')) {
        $costs = $costs->where('created_by', app('auth')->user()->id);
      }

      $costs = $costs->sum('amount');
      
      $arr[$day] = (float) $costs;
      $last_week->addDay();
    }
    return response()->json($arr);
  }

  public function monthly(Request $request) {
    $now = Carbon::now()->endOfMonth();
    $last_week = Carbon::now()->startOfMonth();

    $arr = [];
    while($last_week->lte($now)) {
      // info($last_week->format('D'));
      $day = $last_week->format('Y-m-d');

      $costs = CampaignCost::where('cost_date', $last_week->format('Y-m-d'))
                ->where('is_confirmed', true);

      if(app('auth')->user()->hasRole('Marketing')) {
        $costs = $costs->where('created_by', app('auth')->user()->id);
      }

      $costs = $costs->sum('amount');
      
      $arr[$day] = (float) $costs;
      $last_week->addDay();
    }
    return response()->json($arr);
  }

  public function yearly(Request $request) {
    $now = Carbon::now()->endOfYear();
    $last_week = Carbon::now()->startOfYear();

    $arr = [];
    while($last_week->lte($now)) {
      // info($last_week->format('D'));
      $day = $last_week->format('M');

      $costs = CampaignCost::where('cost_date', 'like', $last_week->format('Y-m').'%')
                ->where('is_confirmed', true);

      if(app('auth')->user()->hasRole('Marketing')) {
        $costs = $costs->where('created_by', app('auth')->user()->id);
      }

      $costs = $costs->sum('amount');
      
      $arr[$day] = (float) $costs;
      $last_week->addMonth();
    }
    return response()->json($arr);
  }

  public function store(Request $request) {
    $params = $request->all();
    $campaign = Campaign::findOrFail($params['campaign_id']);
    $rules = CampaignCost::$rules;
    $rules['cost_date'] = [
      'required', 
      Rule::unique('campaign_costs')->where(function ($query) use ($params) {
        return $query->where('campaign_id', $params['campaign_id'] )
                ->where('cost_date', $params['cost_date']);
      })
    ];
    $this->validate($request, $rules);
    $params['created_by'] = app('auth')->user()->id;

    $data = CampaignCost::create($params);
    return response()->json($data, 201);
  }

  public function show($id) {
    return response()->json(CampaignCost::with(['campaign', 'creator'])->findOrFail($id));
  }

  public function update($id, Request $request) {
    $data = CampaignCost::findOrFail($id);

    if(!app('auth')->user()->hasRole('Admin')) {
      if ($data->created_by != app('auth')->user()->id) {
        return response()->json(['error' => 'Unauthorized - Cannot change other\'s data.'], 401);
      }
    } 
    $user_id = app('auth')->user()->hasRole('Admin') ? $data->id : app('auth')->user()->id;

    $rules = CampaignCost::$rules;
    $rules['cost_date'] = [
      'required', 
      Rule::unique('campaign_costs')->where(function ($query) use ($id, $data) {
        return $query->where('id', '<>', $id)
                    ->where('campaign_id', $data->campaign_id)
                    ->where('cost_date', $data->cost_date);
      })
    ];
    $this->validate($request, $rules);
    $params = $request->all();
    $data->update($params);
    return response()->json($data, 200);
  }

  public function destroy($id) {
    $data = CampaignCost::with(['campaign', 'creator'])->findOrFail($id);

    if(!app('auth')->user()->hasRole('Admin')) {
      if($data->created_by != app('auth')->user()->id) {
        return response()->json(['message' => 'Unauthorized - Cannot delete other\'s data.'], 401);
      }
    }
    if($data->is_confirmed) {
      return response()->json(['message' => 'Unable to delete already confirmed data'], 403);
    }
    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }
}