<?php

namespace App\Http\Controllers;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\Campaign;
use App\CampaignCost;
use App\CampaignLead;
use App\Sale;

class CampaignController extends Controller
{

  public function index() {
    if(app('auth')->user()->hasRole('Admin')) {
      $data = Campaign::with(['product', 'ad_type', 'creator',  'costs', 'leads'])->orderBy('created_at', 'desc')->get();
    } else {
      $data = Campaign::where('created_by', app('auth')->user()->id)
        ->with(['product', 'ad_type', 'creator', 'costs', 'leads'])->orderBy('created_at', 'desc')->get();
    }
    return response()->json($data);
  }

  public function store(Request $request) {
    $rules = Campaign::$rules;
    $rules['code'] = [
      'required', 
      Rule::unique('campaigns')->where(function ($query) {
        return $query->where('created_by', app('auth')->user()->id);
      })
    ];
    $this->validate($request, $rules);
    $params = $request->all();
    $params['created_by'] = app('auth')->user()->id;

    if($request->hasFile('screenshot')) {
      $image = $request->file('screenshot');
      $name = time().'.'.$image->getClientOriginalExtension();
      $destinationPath = storage_path('/images');
      if($image->move($destinationPath, $name)) {
        $params['screenshot'] = '/images/'.$name;
      }
    }

    $data = Campaign::create($params);
    return response()->json($data, 201);
  }

  public function show($id) {
    return response()->json(Campaign::with(['product', 'ad_type', 'creator', 'costs', 'leads'])->findOrFail($id));
  }

  public function summary(Request $request) {
    if(app('auth')->user()->hasRole('Admin')) { 
      $campaigns = Campaign::with(['creator', 'product'])->get();
    }
    if(app('auth')->user()->hasRole('Marketing')) {
      $campaigns = Campaign::where('created_by', app('auth')->user()->id)->with(['creator', 'product'])->get();
    }
    foreach($campaigns as $campaign) {
      $cost = CampaignCost::where('campaign_id', $campaign->id);
      $leads = CampaignLead::where('campaign_id', $campaign->id);
      $sales = Sale::where('campaign_id', $campaign->id)->where('call_status', 'c');
      if($request->query('from_date') && $request->query('end_date')) {
        $cost = $cost->whereBetween('cost_date', [$request->query('from_date'), $request->query('end_date')]);
        $leads = $leads->whereBetween('assigned_date', [$request->query('from_date'), $request->query('end_date')]);
        $sales = $sales->whereBetween('sales_date', [$request->query('from_date'), $request->query('end_date')]);
      }
      $leads = $leads->get();
      $sales = $sales->get();
      $total_qty = 0;
      $total_sales = 0;
      foreach($sales as $sale){
        $total_qty += $sale->qty;
        $total_sales += $sale->total_sales;
      }
      $campaign->total_cost = (int) $cost->sum('amount');
      $campaign->total_leads = count($leads);
      $campaign->total_qty = $total_qty;
      $campaign->total_sales = $total_sales;
    }
    return response()->json($campaigns);
  }

  public function update(Request $request, $id) {
    $data = Campaign::findOrFail($id);

    if(!app('auth')->user()->hasRole('Admin')) {
      if ($data->created_by != app('auth')->user()->id) {
        return response()->json(['error' => 'Unauthorized - Cannot change other\'s data.'], 401);
      }
    } 
    $user_id = app('auth')->user()->hasRole('Admin') ? $data->id : app('auth')->user()->id;

    // die($request->input['name']);
    $rules = Campaign::$rules;
    $rules['code'] = [
      'required', 
      Rule::unique('campaigns')->where(function ($query) use ($id, $user_id) {
        return $query->where('id', '<>', $id)
                    ->where('created_by', $user_id);
      })
    ];
    $this->validate($request, $rules);
    $params = $request->all();

    if($request->hasFile('screenshot')) {
      $image = $request->file('screenshot');
      $name = time().'.'.$image->getClientOriginalExtension();
      $destinationPath = storage_path('/images');
      if($image->move($destinationPath, $name)) {
        $params['screenshot'] = '/images/'.$name;
      }
    }

    $data->update($params);
    return response()->json($data, 200);
  }

  public function destroy($id) {
    $data = Campaign::with(['product', 'ad_type', 'creator', 'costs', 'leads'])->findOrFail($id);

    if(!app('auth')->user()->hasRole('Admin')) {
      if($data->created_by != app('auth')->user()->id) {
        return response()->json(['message' => 'Unauthorized - Cannot delete other\'s data.'], 401);
      }
    }
    if(count($data->costs) > 0 || count($data->leads) > 0) {
      return response()->json(['message' => 'Unable to delete - Other data linked to this data.'], 403);
    }
    $data->delete();
    return response()->json(['message' => 'Delete Successfully'], 200);
  }
}