<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Mail\RegisterConfirmation;
use App\Mail\ForgotPassword;
use App\Http\Controllers\Controller;
use App\Role;
use App\User;

class AuthController extends Controller
{
    /**
     * Create a new AuthController instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth', ['except' => ['login', 'register', 'confirm_email']]);
    }

    /**
     * Get a JWT via given credentials.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(Request $request)
    {
        $credentials = $request->only(['email', 'password']);

        if (! $token = app('auth')->setTTL(60 * 24 * 7)->attempt($credentials)) {
            return response()->json(['message' => 'Invalid email or password.'], 401);
        }

        return $this->respondWithToken($token);
    }

    /**
     * Get the authenticated User.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function me()
    {
        return response()->json([
            "user"  => app('auth')->user(),
            "role"  => app('auth')->user()->getRoleNames()[0]
          ]);
    }

    /**
     * Log the user out (Invalidate the token).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout()
    {
        app('auth')->logout();

        return response()->json(['message' => 'Successfully logged out']);
    }

    /**
     * Refresh a token.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function refresh()
    {
        return $this->respondWithToken(app('auth')->refresh());
    }

    public function change_password(Request $request) {
      if (!(Hash::check($request->get('current_password'), app('auth')->user()->password))) {
          // The passwords matches
          return response()->json(["message" => "Your current password does not matches with the password you provided. Please try again."], 403);
      }
      if(strcmp($request->get('current_password'), $request->get('new_password')) == 0){
          //Current password and new password are same
          return response()->json(["message" => "New Password cannot be same as your current password. Please choose a different password."], 403);
      }
      $this->validate($request, [
          'current_password' => 'required',
          'new_password' => 'required|string|min:6|confirmed',
      ]);

      $user = app('auth')->user();
      $user->password = password_hash($request->get('new_password'), PASSWORD_DEFAULT);
      $user->save();

      return response()->json(['message' => 'Password changed successfully']);
    }

    public function register(Request $request) {
      $role = Role::where('name', 'User')->first();

      if (!$role) {
          return response()->json(['error' => 'Invalid role.'], 403);
      }

      $rules = User::$rules;
      $request['roles'] = $role->id;
      $request['username'] = $request['email'];
      $this->validate($request, $rules);

      $params = $request->all();
      $params['password'] = password_hash($params['password'], PASSWORD_DEFAULT);
      $params['status'] = false;
      $params['verified'] = false;
      $params['token'] = sha1(time());

      $data = User::create($params);
      $data->assignRole($role->name);

      Mail::to($data->email)->send(new RegisterConfirmation($data));

      return response()->json($data, 201);
  }

  public function test_email() {
      Mail::to('saifi.ikmal@gmail.com')->send(new RegisterConfirmation('12345'));
      return response()->json('E-mail has been sent', 200);
  }

  public function confirm_email(Request $request) {

      $email = $request->query('email');
      $token = $request->query('token');

      if (!isset($email) || $email == '' || !isset($token) || $token == '') {
          return response()->json(['message' => 'Missing parameters.'], 403);
      }
      $user = User::where('token', $request->query('token'))
              ->where('email', $request->query('email'))->first();
      if (!$user) {
          return response()->json(['message' => 'Invalid token.'], 403);
      }

      if ($user->verified) {
          return response()->json(['message' => 'E-mail already verified.'], 403);
      }
      $user->update([
          'status' => true,
          'verified' => true,
          'token' => null
      ]);

      return response()->json(['message' => 'E-mail verified succesfully.']);
  }

  public function forgot_password(Request $request) {
      $params = $request->all();

      if (!isset($params['email']) || $params['email'] == '') {
          return response()->json(['message' => 'Missing e-mail'], 403);
      }

      $user = User::whereRaw('LOWER(`email`) = ?', $params['email'])->first();

      if (!$user) {
          return response()->json(['message' => 'Cannot found user with this e-mail'], 403);
      }

      if (!$user->status || !$user->verified) {
          return response()->json(['message' => 'User account is not yet verified/enabled'], 403);
      }

      $forgot_password_token = sha1(time());
      $user->update([
          // 'password' => '',
          'forgot_password_token' => $forgot_password_token,
      ]);

      Mail::to($user->email)->send(new ForgotPassword($user));

      return response()->json(['message' => 'Reset password e-mail has been sent.']);
  }

  public function reset_password(Request $request) {
      $params = $request->all();

      if (!isset($params['email']) || $params['email'] == '' || !isset($params['token']) || $params['token'] == ''
          || !isset($params['password']) || $params['password'] == '') {
          return response()->json(['message' => 'Missing parameters.'], 403);
      }

      $user = User::where('forgot_password_token', $params['token'])
              ->where('email', $params['email'])->first();
      if (!$user) {
          return response()->json(['message' => 'Invalid token.'], 403);
      }

      if (!$user->verified || !$user->status) {
          return response()->json(['message' => 'User account is not yet verified/enabled'], 403);
      }
      $user->update([
          'password' => password_hash($params['password'], PASSWORD_DEFAULT),
          'forgot_password_token' => null
      ]);

      return response()->json(['message' => 'Password reset succesfully.']);
  }

    /**
     * Get the token array structure.
     *
     * @param  string $token
     *
     * @return \Illuminate\Http\JsonResponse
     */
    protected function respondWithToken($token)
    {
        return response()->json([
            'user'  => app('auth')->user(),
            'role'  => app('auth')->user()->getRoleNames()[0],
            'access_token' => $token,
            'token_type' => 'bearer',
            'expires_in' => app('auth')->factory()->getTTL() * 60
        ]);
    }
}